package com.shipsy.ondemand.riderapp.domain.usecase.autoswipe

import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.domain.orderstatus.OrderType
import com.shipsy.ondemand.riderapp.framework.network.model.login.GeoConfigModel
import com.shipsy.ondemand.riderapp.framework.network.model.login.Location
import com.shipsy.ondemand.riderapp.interactor.data.LocationData
import com.shipsy.ondemand.riderapp.interactor.usecase.autoswipe.AutoSwipeUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.autoswipe.CheckGoingForPickupAutoSwipe
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.*
import com.shipsy.ondemand.riderapp.interactor.usecase.location.CalculateDistanceBetweenLocationUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.IsRiderCheckedIn
import com.shipsy.ondemand.riderapp.shared.cache.CurrentOrders

class AutoSwipeUseCaseImpl(
    private val localStore: LocalStore,
    private val isRiderCheckedIn: IsRiderCheckedIn,
    private val appDatabase: Database,
    private val getGeoConfigUseCase: GetGeoConfigUseCase,
    private val calculateDistanceBetweenLocationUseCase: CalculateDistanceBetweenLocationUseCase,
    private val returnToStoreEventUseCase: ReturnToStoreEventUseCase,
    private val pickingProcessStartedEventUseCase: PickingProcessStartedEventUseCase,
    private val startBikeEventUseCase: StartBikeEventUseCase,
    private val reachGateEventUseCase: ReachGateEventUseCase,
    private val checkGoingForPickupAutoSwipe: CheckGoingForPickupAutoSwipe,
    private val goingForPickupEventUseCase: GoingForPickupEventUseCase,
) : AutoSwipeUseCase {

    private val currentRiderStatus: String
        get() = localStore
            .getValue(RIDER_STATE_RESPONSE_WORKER_STATUS, "")
    private val hubLat: Double
        get() = localStore.getValue(HUB_LAT, 0.0)
    private val hubLng: Double
        get() = localStore.getValue(HUB_LNG, 0.0)

    override suspend fun invoke(input: AutoSwipeUseCase.Input): Any? {
        try {
            val isCheckedIn = isRiderCheckedIn.invoke()
            //Stop if user is off duty or not logged in.
            if (!isCheckedIn) {
                return null
            }
            if (input.location.lat != 0.0 && input.location.lng != 0.0 && isCheckedIn) {
                when (currentRiderStatus.lowercase()) {
                    WorkerStatusType.RETURNING_TO_STORE.lowercase() -> {
                        autoSwipeForBackToStore(
                            input.location
                        )
                    }
                    WorkerStatusType.IN_STORE.lowercase() -> inStoreRiderAction(
                        input.location
                    )
                    WorkerStatusType.OUT_FOR_DELIVERY.lowercase() -> outForDeliveryRiderAction(
                        input.location
                    )
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return null
    }

    private suspend fun inStoreRiderAction(location: LocationData) {
        checkGoingForPickupAutoSwipe(location)

        val isPickProcessFlowEnabled =
            localStore.getValue(PICKUP_CONFIG_ENABLE_PICKING_FLOW, false)
        val isPudoFlowEnabled =
            localStore.getValue(SHARED_PREF_ENABLE_PICKUP_DELIVERY, false)
        if (isPudoFlowEnabled)
            return
        return when {
            isPickProcessFlowEnabled -> autoSwipeForPickingUpStarted(location)
            else -> autoSwipeForStartBike(location)
        }
    }

    private suspend fun autoSwipeForPickingUpStarted(location: LocationData) {
        if (validateForPickingProcessStarted(location)) {
            val orders = findPickupOrdersByStatus(OrderStatusType.ASSIGNED_FOR_DELIVERY)

            val referenceNumbers = orders?.map { it.reference_number }
            if (referenceNumbers.isNullOrEmpty()) return

            try {
                if (checkEventExists(
                        RiderStateEventType.PICKING_PROCESS_STARTED, referenceNumbers
                    )
                ) {
                    return
                }
                pickingProcessStartedEventUseCase.invoke(
                    PickingProcessStartedEventUseCase.Input(
                        referenceNumbers = referenceNumbers,
                        syncType = SyncType.Sync,
                        autoSwipe = true
                    )
                )
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    private suspend fun autoSwipeForStartBike(location: LocationData) {
        if (validateForStartBike(location)) {
            val orders = appDatabase.currentOrdersDao
                .getOrderNotByStatus(listOf(OrderStatusType.DELIVERED))

            val referenceNumbers = orders?.map { it.reference_number }
            if (referenceNumbers.isNullOrEmpty()) return

            try {
                if (checkEventExists(RiderStateEventType.START_BIKE, referenceNumbers)) {
                    return
                }
                startBikeEventUseCase.invoke(
                    StartBikeEventUseCase.Input(
                        referenceNumbers = referenceNumbers,
                        syncType = SyncType.Sync,
                        autoSwipe = true
                    )
                )

            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    private suspend fun validateForPickingProcessStarted(location: LocationData): Boolean {
        val autoSwipeConfig: GeoConfigModel =
            getAutoSwipeConfig(GeofenceConfigType.PICKING_UP_STARTED)
        if (!autoSwipeConfig.auto_swipe) return false

        val geofenceDistance = autoSwipeConfig.geofence_distance ?: return false

        val distance = findDistanceFromHub(location)
        return distance < geofenceDistance
    }

    private suspend fun validateForStartBike(location: LocationData): Boolean {
        val autoSwipeConfig: GeoConfigModel =
            getAutoSwipeConfig(GeofenceConfigType.PICKING_UP_STARTED)
        if (!autoSwipeConfig.auto_swipe) return false

        val geofenceDistance = autoSwipeConfig.geofence_distance ?: return false

        val distance = findDistanceFromHub(location)
        return distance > (autoSwipeConfig.auto_swipe_distance ?: 0) && distance < geofenceDistance
    }

    private suspend fun outForDeliveryRiderAction(location: LocationData) {
        checkGoingForPickupAutoSwipe(location)

        val isPickProcessFlowEnabled =
            localStore.getValue(PICKUP_CONFIG_ENABLE_PICKING_FLOW, false)
        return when {
            isPickProcessFlowEnabled -> {
                autoSwipeForPudoFlow(location)
                autoSwipeForReachGate(location)
            }
            else -> autoSwipeForReachGate(location)
        }
    }

    private suspend fun checkGoingForPickupAutoSwipe(location: LocationData) {
        checkGoingForPickupAutoSwipe.invoke(
            CheckGoingForPickupAutoSwipe.Input(
                location,
                ::performGoingForPickup
            )
        )
    }

    private suspend fun autoSwipeForReachGate(location: LocationData) {
        val autoSwipeConfig = getAutoSwipeConfig(GeofenceConfigType.REACHED_GATE)
        if (autoSwipeConfig.auto_swipe) {
            markReachGate(autoSwipeConfig, location)
        }
    }

    private suspend fun markReachGate(
        reachedAtGateConfig: GeoConfigModel,
        location: LocationData
    ) {
        val referenceNumbers = findDeliveryOrdersByStatus(
            listOf(
                OrderStatusType.START_BIKE,
                OrderStatusType.PICKUP_COMPLETED
            )
        )
            ?.filter {
                if (it.location != null) {
                    val distance = findDistance(it.location, location)
                    println("reachedAtGateConfig.auto_swipe_distance: ${reachedAtGateConfig.auto_swipe_distance}")
                    distance < (reachedAtGateConfig.auto_swipe_distance ?: 0)
                } else {
                    false
                }
            }
            ?.map { it.reference_number }

        if (!referenceNumbers.isNullOrEmpty()) {
            reachGateEventUseCase.invoke(
                ReachGateEventUseCase.Input(
                    referenceNumbers,
                    SyncType.Sync,
                    autoSwipe = true
                )
            )
        }
    }

    private suspend fun autoSwipeForPudoFlow(location: LocationData) {
        val autoSwipeConfig = getAutoSwipeConfig(GeofenceConfigType.PICKING_UP_STARTED)
        if (autoSwipeConfig.auto_swipe) {
            markPickUpStartedPudoFlow(autoSwipeConfig, location)
        }
    }

    private suspend fun markPickUpStartedPudoFlow(
        autoSwipeConfig: GeoConfigModel,
        location: LocationData
    ) {
        val referenceNumbers = findPickupOrdersByStatus(OrderStatusType.PICKED_UP)
            ?.filter {
                if (it.location != null) {
                    val distance = findDistance(it.location, location)
                    distance < (autoSwipeConfig.auto_swipe_distance ?: 0)
                } else {
                    false
                }
            }
            ?.map { it.reference_number }

        if (referenceNumbers.isNullOrEmpty()) return

        try {
            if (checkEventExists(RiderStateEventType.PICKING_PROCESS_STARTED, referenceNumbers)) {
                return
            }
            pickingProcessStartedEventUseCase.invoke(
                PickingProcessStartedEventUseCase.Input(
                    referenceNumbers = referenceNumbers,
                    syncType = SyncType.Sync,
                    autoSwipe = true
                )
            )
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private suspend fun findPickupOrdersByStatus(status: String): List<CurrentOrders>? {
        return appDatabase.currentOrdersDao.getCurrentOrderByStatusAndTaskType(
            OrderType.PICKUP,
            listOf(status)
        )
    }

    private suspend fun findDeliveryOrdersByStatus(status: List<String>): List<CurrentOrders>? {
        return appDatabase.currentOrdersDao.getCurrentOrderByStatusAndTaskType(
            OrderType.DELIVERY,
            status
        )
    }

    private suspend fun findDistance(location: Location, location2: LocationData): Double {
        val input = CalculateDistanceBetweenLocationUseCase.Input(
            location.lat ?: 0.0, location.lng ?: 0.0,
            location2.lat, location2.lng,
        )
        return calculateDistanceBetweenLocationUseCase.invoke(input).distance
    }

    private suspend fun findDistanceFromHub(location: LocationData) =
        calculateDistanceBetweenLocationUseCase.invoke(
            CalculateDistanceBetweenLocationUseCase.Input(
                hubLat,
                hubLng,
                location.lat,
                location.lng
            )
        ).distance

    private suspend fun getAutoSwipeConfig(type: String): GeoConfigModel {
        return getGeoConfigUseCase.invoke(GetGeoConfigUseCase.Input(type)).geoConfigModel
    }

    private suspend fun autoSwipeForBackToStore(location: LocationData) {
        val autoSwipeConfig =
            getGeoConfigUseCase.invoke(GetGeoConfigUseCase.Input(GeofenceConfigType.BACK_TO_STORE)).geoConfigModel
        if (validateRiderForBackToStore(autoSwipeConfig, location)) {
            markRiderBackToStore()
        }
    }

    private suspend fun markRiderBackToStore() {
        try {
            val status = listOf(OrderStatusType.DELIVERED, OrderStatusType.ATTEMPTED)
            val orders = appDatabase.currentOrdersDao.getOrdersByStatus(status)
            val referenceNumbers = orders?.map { it.reference_number }

            if (referenceNumbers.isNullOrEmpty()) return
            if (checkEventExists(RiderStateEventType.RETURN_TO_HUB, referenceNumbers)) return

            returnToStoreEventUseCase.invoke(
                ReturnToStoreEventUseCase.Input(
                    referenceNumbers,
                    SyncType.Sync,
                    true
                )
            )
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private suspend fun checkEventExists(event: String, referenceNumbers: List<String>): Boolean {
        val riderStateEventDao = appDatabase.riderStateEventDao
        val riderStateEventArrayList = riderStateEventDao.getRiderStateEventDataItems()
        for (events in riderStateEventArrayList) {
            if (events.type == event &&
                events.reference_number_array.containsAll(referenceNumbers)
            )
                return true
        }
        return false
    }


    private suspend fun validateRiderForBackToStore(
        autoSwipeConfig: GeoConfigModel,
        location: LocationData
    ): Boolean {
        val distance = findDistanceFromHub(location)
        return autoSwipeConfig.auto_swipe && distance < (autoSwipeConfig.auto_swipe_distance ?: 0)
    }

    private suspend fun performGoingForPickup(referenceNumber: String) {
        try {
            val referenceNumberList = listOf(referenceNumber)

            if (checkEventExists(RiderStateEventType.GOING_FOR_PICKUP, referenceNumberList))
                return

            goingForPickupEventUseCase.invoke(
                GoingForPickupEventUseCase.Input(
                    referenceNumber,
                    SyncType.Sync,
                    true
                )
            )
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }
}