package com.shipsy.ondemand.riderapp.domain.usecase

import com.shipsy.ondemand.riderapp.firebase.FirebaseEventHandler
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.framework.network.getResult
import com.shipsy.ondemand.riderapp.framework.network.model.UserLogin
import com.shipsy.ondemand.riderapp.framework.network.model.login.LoginResponse
import com.shipsy.ondemand.riderapp.interactor.repository.RiderLoginRepository
import com.shipsy.ondemand.riderapp.interactor.usecase.*

class LoginRiderUseCaseImpl(
    private val repo: RiderLoginRepository,
    private val saveLoginDataUseCase: SaveLoginDataUseCase,
    private val saveCreds: SaveUserCredentialsUseCase
) : LoginRiderUseCase {
    override suspend fun invoke(input: UserLogin): UseCaseResult<Any?> {
        return when {
            input.username.trim().isEmpty() -> return UseCaseResult.failure(
                ErrorData.GeneralError(
                    displayType = DisplayType.SnackBar("Username can't be empty")
                )
            )
            input.password.trim().isEmpty() -> return UseCaseResult.failure(
                ErrorData.GeneralError(
                    displayType = DisplayType.SnackBar( "Password can't be empty")
                )
            )
            else -> login(input)
        }
    }

    private suspend fun login(input: UserLogin): UseCaseResult<LoginResponse> {
        saveUserCreds(input)

        val response = getResult<LoginResponse> { repo.loginRider(input) }
        if (response.isSuccess) {
            saveLoginDataUseCase.invoke(response.getOrNull()!!)

            FirebaseEventHandler.initializeFirebase()
            FirebaseEventHandler.setUserPropertyUpdated(false)
            FirebaseEventHandler.updateUserDetailAfterLogin()
        }
        return response
    }

    private fun saveUserCreds(input: UserLogin) {
        try {
            val credsInput = if (input.rememberCreds == true) {
                SaveUserCredentialsUseCase.Input(input.username, input.password)
            } else {
                SaveUserCredentialsUseCase.Input("", "")
            }
            saveCreds.invoke(credsInput)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }
}