package com.shipsy.ondemand.riderapp.domain.usecase

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.firebase.FirebaseEventHandler
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.framework.network.getResult
import com.shipsy.ondemand.riderapp.framework.network.model.checkin.CheckInRequest
import com.shipsy.ondemand.riderapp.framework.network.model.login.ChildHubModel
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.interactor.data.LocationData
import com.shipsy.ondemand.riderapp.interactor.repository.CheckInRepository
import com.shipsy.ondemand.riderapp.interactor.usecase.CheckInRiderUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.UpdateRiderStateUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.CalculateDistanceBetweenLocationUseCase
import kotlinx.serialization.builtins.ListSerializer
import kotlinx.serialization.json.Json

class CheckInRiderUseCaseImpl(
    private val repo: CheckInRepository, private val localStore: LocalStore,
    private val calculateDistanceBetweenLocationUseCase: CalculateDistanceBetweenLocationUseCase,
    private val updateRiderStateUseCase: UpdateRiderStateUseCase
) :
    CheckInRiderUseCase {

    private val hubLat: Double
        get() = localStore.getValue(HUB_LAT, 0.0)
    private val hubLng: Double
        get() = localStore.getValue(HUB_LNG, 0.0)

    override suspend fun invoke(input: CheckInRiderUseCase.Input): UseCaseResult<Boolean> {
        val result = checkInRiderWithLocationCheck(LocationData(input.lat, input.lng, 0.0))
        val canCheckIn = result.first
        return if(canCheckIn) {
            FirebaseEventHandler.logCheckInEvent(input.lat, input.lng, result.second)
            proceedToCheckIn(input)
        } else {
            UseCaseResult.failure(
                ErrorData.GeneralError(
                    displayType = DisplayType.Alert("Alert", "You need to be close to the store to CheckIn")
                )
            )
        }
    }

    private suspend fun proceedToCheckIn(input: CheckInRiderUseCase.Input): UseCaseResult<Boolean> {
        val checkInRequest = CheckInRequest(
            lat = input.lat,
            lng = input.lng,
            mobile_time = DateTime.currentTimeMillis().toString()
        )
        val response =
            getResult<RiderStateResponse> { repo.checkInRider(getWorkerId(), checkInRequest) }
        if (response.isSuccess) {
            updateRiderStateUseCase.invoke(response.getOrNull())
            return UseCaseResult.success(true)
        }

        return UseCaseResult.failure(response.getErrorData()!!)
    }


    private suspend fun checkInRiderWithLocationCheck(location: LocationData): Pair<Boolean, Double> {
        val thresholdDistance: Long = localStore.getValue(
            SHARED_PREF_ON_DEMAND_CHECK_IN_DISTANCE,
            STATUS_SEND_DISTANCE_FALLBACK
        )
        val hubLocation = LocationData(hubLat, hubLng, 0.0)
        val isCheckInForChildAllowed: Boolean = localStore.getValue(
            SHARED_PREF_ALLOW_CHECKIN_FROM_CHILD_HUBS,
            false
        )
        println("thresholdDistance: $thresholdDistance hubLocation: $hubLocation isCheckInForChildAllowed: $isCheckInForChildAllowed")
        val totalDistance = if (isCheckInForChildAllowed) {
            getDistanceFromNearestHub(
                location.lat,
                location.lng,
                thresholdDistance.toDouble()
            )
        } else {
            findDistance(location, hubLocation)
        }
        val canCheckIn = totalDistance < thresholdDistance
        println("totalDistance $totalDistance < thresholdDistance $thresholdDistance, $canCheckIn")
        return Pair(canCheckIn, totalDistance)
    }



    private suspend fun findDistance(location: LocationData, location2: LocationData): Double {
        println("location.lat ${location.lat}, location.lng ${location.lng}, location2.lat ${location2.lat}, location2.lng ${location2.lng}")
        val input = CalculateDistanceBetweenLocationUseCase.Input(
            location.lat, location.lng,
            location2.lat, location2.lng,
        )
        return calculateDistanceBetweenLocationUseCase.invoke(input).distance
    }

    private fun getWorkerId(): String {
        return localStore.getValue(rider_id, "")
    }

    private suspend fun getDistanceFromNearestHub(
        lat: Double,
        lng: Double,
        thresholdDistance: Double
    ): Double {
        val savedChildHubs = localStore.getValue(
            SHARED_PREF_CHILD_HUB_LIST,
            ""
        )
        println("savedChildHubs: $savedChildHubs")
        var distance = Int.MAX_VALUE.toDouble()
        if (savedChildHubs.isEmpty()) return distance
        val childHubList: List<ChildHubModel> =
            try {
                Json.decodeFromString(ListSerializer(ChildHubModel.serializer()), savedChildHubs)
            } catch (e: Exception) {
                return distance
            }

        println("childHubList.size: " + childHubList.size)
        val location = LocationData(lat, lng, 0.0)
        for (hub in childHubList) {
            val hubLocation = LocationData(hub.lat ?: 0.0, hub.lng ?: 0.0, 0.0)
            if (hubLocation.isValid) {
                val temp: Double = findDistance(location, hubLocation)
                if (temp < distance) {
                    distance = temp
                    if (temp <= thresholdDistance) {
                        return temp
                    }
                }
            }
        }
        return distance
    }
}