package com.shipsy.ondemand.riderapp.cache.dao

import com.shipsy.ondemand.riderapp.firebase.FirebaseEventHandler
import com.shipsy.ondemand.riderapp.framework.network.model.login.CurrentOrder
import com.shipsy.ondemand.riderapp.shared.cache.AppDatabaseQueries
import com.shipsy.ondemand.riderapp.shared.cache.CurrentOrders

class CurrentOrdersDao(private val dbQueries: AppDatabaseQueries) {

    fun insertCurrentOrders(currentOrders: List<CurrentOrder>?) {
        dbQueries.transaction {
            dbQueries.removeAllCurrentOrders()
            currentOrders?.forEach {
                dbQueries.insertCurrentOrders(
                    id = null,
                    reference_number = it.reference_number,
                    status = it.status,
                    task_type = it.task_type,
                    is_cod = it.is_cod,
                    cod_amount = it.cod_amount,
                    is_vip = it.is_vip,
                    is_bulk = it.is_bulk,
                    notes = it.notes,
                    payment_details = it.payment_details,
                    pieces_details = it.pieces_detail,
                    order_number = it.order_number,
                    order_type = it.order_type,
                    otp = it.otp,
                    makeline_time = it.makeline_time,
                    final_amount = it.final_amount,
                    order_promise_start_time_ondemand = it.order_promise_start_time_ondemand,
                    ondemand_delivery_promise_mins = it.ondemand_delivery_promise_mins,
                    priority = it.priority?.toLong(),
                    ondemand_discount = it.ondemand_discount,
                    lastUpdatedAt = it.last_updated_at,
                    assignTime = it.assign_time,
                    storageLocation = it.storage_location,
                    huDetails = it.hu_details,
                    location = it.location,
                    tip = it.tip,
                    eligible_payout = it.eligible_payout,
                    order_instructions = it.order_instructions
                )
            }
        }
    }

    suspend fun getCurrentOrders(): List<CurrentOrders> {
        val currentOrders: List<CurrentOrders> = dbQueries.transactionWithResult {
            dbQueries.getAllCurrentOrders().executeAsList()
        }
        return currentOrders
    }

    suspend fun getOrdersByReferenceNumber(referenceNumber: String): CurrentOrder? {
        val order: CurrentOrders? = dbQueries.transactionWithResult {
            dbQueries.getOrdersByReferenceNumber(referenceNumber).executeAsOneOrNull()
        }
        return order?.mapToCurrentOrder()
    }

    suspend fun getOrdersByReferenceNumbers(referenceNumbers: List<String>): List<CurrentOrder>? {
        val order: List<CurrentOrders>? = dbQueries.transactionWithResult {
            dbQueries.getOrdersByReferenceNumbers(referenceNumbers).executeAsList()
        }
        return order?.map { it.mapToCurrentOrder() }
    }

    suspend fun getReferenceNumbersByOrderNumbers(orderNumbers: List<String>): List<String>? {
        return dbQueries.transactionWithResult {
            dbQueries.getReferenceNumbersByOrderNumbers(orderNumbers).executeAsList()
        }
    }

    suspend fun getOrdersByStatus(status: List<String>): List<CurrentOrders>? {
        val order: List<CurrentOrders>? = dbQueries.transactionWithResult {
            dbQueries.getCurrentOrdersByStatus(status).executeAsList()
        }
        return order
    }

    suspend fun updateOrderStatusByReferenceNumber(referenceNumber: List<String>, status: String) {
        referenceNumber.forEach {
            FirebaseEventHandler.logOrderStateEvent(status, it)
        }
        dbQueries.updateOrderStatusByReferenceNumber(status, referenceNumber)
    }

    suspend fun getOrderNotByStatus(status: List<String>): List<CurrentOrders>?  {
        val order: List<CurrentOrders>? = dbQueries.transactionWithResult {
            dbQueries.currentOrdersByNotStatus(status).executeAsList()
        }
        return order
    }

    suspend fun getCurrentOrderByStatusAndTaskType(taskType:String,status: List<String>): List<CurrentOrders>?  {
        val order: List<CurrentOrders>? = dbQueries.transactionWithResult {
            dbQueries.getCurrentOrderByStatusAndTaskType(status,taskType).executeAsList()
        }
        return order
    }

    suspend fun deleteOrderByReferenceNumber(referenceNumber: String) {
        dbQueries.deleteOrderByReferenceNumber(referenceNumber)
    }

    suspend fun deleteOrderByOrderNumber(orderNumber: String) {
        dbQueries.deleteOrderByOrderNumber(orderNumber)
    }
}