package com.shipsy.ondemand.riderapp.framework.mixpanel

import android.content.Context
import com.mixpanel.android.mpmetrics.MixpanelAPI
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.HUB_CODE
import com.shipsy.ondemand.riderapp.domain.const.ORG_ID
import com.shipsy.ondemand.riderapp.domain.const.PREF_SDK_MIXPANEL_PROJECT_TOKEN
import com.shipsy.ondemand.riderapp.domain.const.rider_id
import com.shipsy.ondemand.riderapp.domain.const.worker_code
import com.shipsy.ondemand.riderapp.interactor.usecase.MixpanelEventHandlerUseCase
import org.json.JSONObject
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject
private const val TRY_CATCH_LOG: String = "TRY_CATCH_LOG"
private const val EVENT_LOG: String = "EVENT_LOG"
private const val ARG_TYPE: String = "type"

internal class MixpanelEventHandlerNativeImpl : KoinComponent, MixpanelEventHandlerUseCase {

    private val context: Context by inject()
    private val localStore : LocalStore by inject()
    private val mixpanelToken
        get() = localStore.getValue(PREF_SDK_MIXPANEL_PROJECT_TOKEN, "")


    val mixpanelAPI: MixpanelAPI?
        get() {
            if (mixpanelToken.isNullOrBlank()) return null
            val api by lazy { MixpanelAPI.getInstance(context, mixpanelToken, true) }
            api.track("Mixpanel Initialized")
            return api
        }

    override fun setupUser() {
        val riderId = localStore.getValue(rider_id, "")
        val workerCode = localStore.getValue(worker_code, "")
        val hubCode = localStore.getValue(HUB_CODE, "")
        val orgId = localStore.getValue(ORG_ID, "")
        mixpanelAPI?.identify(riderId)
        mixpanelAPI?.people?.set("worker_code", workerCode)
        mixpanelAPI?.people?.set("hub_code", hubCode)
        mixpanelAPI?.people?.set("org_id", orgId)

        val map = mutableMapOf(
            "hub_code" to hubCode,
            "worker_code" to workerCode,
            "org_id" to orgId
        )
        trackMultiple("setup_user", map)

        val json = getJsonFromMap(map)

        mixpanelAPI?.registerSuperPropertiesOnce(json)
    }

    private fun getJsonFromMap(map: Map<String, String>): JSONObject {
        val json = JSONObject()
        map.forEach {
            json.put(it.key, it.value)
        }
        return json
    }

    override fun track(event: String, key: String, value: String) {
        if (key.isBlank() || value.isBlank()) mixpanelAPI?.track(event)
        val json = JSONObject()
        json.put(key, value)
        json.put(ARG_TYPE, EVENT_LOG)
        mixpanelAPI?.track(event, json)
    }

    override fun trackMultiple(event: String, map: Map<String, String>) {
        val json = getJsonFromMap(map)
        json.put(ARG_TYPE, EVENT_LOG)
        mixpanelAPI?.track(event, json)
    }

    override fun resetMixpanel() {
        mixpanelAPI?.reset()
        localStore.putValue(PREF_SDK_MIXPANEL_PROJECT_TOKEN, "")
    }

    override fun logCrash(e: Exception) {
        val errorJson = mutableMapOf<String, String>()
        val stackTraceArray = e.stackTraceToString().chunked(250)
        for ((i, stackTrace) in stackTraceArray.withIndex()) {
            errorJson["stackTraceToString$i"] = stackTrace
        }
        println("stackTraceArray: $stackTraceArray , ${stackTraceArray.size}")
        errorJson["stackTraceToString"] = e.stackTraceToString()
        errorJson["message"] = e.message.toString()
        errorJson["cause"] = e.cause.toString()
        errorJson["localizedMessage"] = e.localizedMessage ?: ""
        errorJson["suppressed"] = e.suppressed.toString()
        errorJson["stackTrace"] = e.stackTrace.toString()
        errorJson["exceptionLine"] = e.stackTrace.firstOrNull()?.className + ":" + e.stackTrace.firstOrNull()?.lineNumber
        errorJson[ARG_TYPE] = TRY_CATCH_LOG
        val key =
            "error_" + e.stackTrace.firstOrNull()?.methodName + "_" + e.stackTrace.firstOrNull()?.lineNumber
        trackMultiple(key, errorJson)
    }
}