package com.shipsy.ondemand.core.framework.network

import com.shipsy.ondemand.core.framework.error.PlatformExceptions
import com.shipsy.ondemand.core.framework.util.JsonHelper.convertResponse
import io.ktor.client.call.*
import io.ktor.client.plugins.*

suspend inline fun <reified T> getResult(request: suspend () -> Any?): UseCaseResult<T> {
    return try {
        val result = request.invoke() as T
        UseCaseResult.success(result)
    } catch (error: Exception) {
        extractError(error)
    }
}

suspend fun <T> executeCatching(run: suspend () -> UseCaseResult<T>): UseCaseResult<T> {
    return try {
        run()
    } catch (e: Exception) {
        UseCaseResult.failure(
            ErrorData.GeneralError(
                true,
                DisplayType.SnackBar(e.message ?: "Error processing request!")
            )
        )
    }
}

suspend fun <T> extractError(error: Exception): UseCaseResult<T> {
    println("extractError -> $error")
    return when (error) {
        is ClientRequestException -> {
            try {
                val errorData = error.response.body<NetworkError>().asErrorData()
                println("error data ==> $errorData")
                UseCaseResult.failure(errorData)
            }catch (e: Exception){
                val errorData = error.response.body<String>().convertResponse<NetworkError>().asErrorData()
                UseCaseResult.failure(
                    errorData
                )
            }
        }
        is ServerResponseException -> {
            try {
                val errorData = error.response.body<NetworkError>().asErrorData()
                UseCaseResult.failure(errorData)
            } catch (e: Exception) {
                UseCaseResult.failure(
                    ErrorData.GeneralError(
                        true,
                        DisplayType.SnackBar(error.message)
                    )
                )
            }
        }
        is NoTransformationFoundException ->{
            UseCaseResult.failure(
                ErrorData.GeneralError(
                    true,
                    DisplayType.SnackBar(error.message ?: "something went wrong")
                )
            )
        }
        else -> {
            val errorMessage = PlatformExceptions.getErrorMessage(error) ?: error.message ?: "something went wrong"
            UseCaseResult.failure(
                ErrorData.GeneralError(
                    true,
                    DisplayType.SnackBar(errorMessage)
                )
            )
        }
    }
}