package io.seald.seald_sdk

import kotlinx.coroutines.*

/**
 * The SealdSSKSTmrPlugin class allows to use the SSKS key storage service to store Seald identities
 * easily and securely, encrypted by a key stored on your back-end server.
 *
 * @param apiURL The Seald server for this instance to use. This value is given on your Seald dashboard.
 * @param appId The ID given by the Seald server to your app. This value is given on your Seald dashboard.
 */
class SealdSSKSTmrPlugin constructor(
    apiURL: String,
    appId: String,
) {
    private var mobileSSKSTMR: io.seald.seald_sdk_internals.mobile_sdk.MobileSSKSTMR

    /**
     * @suppress
     */
    init {
        mobileSSKSTMR = io.seald.seald_sdk_internals.mobile_sdk.Mobile_sdk.newSSKSTMRPlugin(apiURL, appId)
    }

    /**
     * Save the Seald account to SSKS.
     *
     * @param sessionId Session ID given by SSKS to your app's server.
     * @param authFactor Authentication method of this user, to which SSKS has sent a challenge at the request of your app's server.
     * @param challenge The challenge sent by SSKS to the user's authentication method. Set to empty string if the SSKS server replies to challenge_send API call with `must_authenticate` set to `false`.
     * @param rawTMRSymKey The raw encryption key used to encrypt / decrypt the stored identity keys. This *MUST* be a cryptographically random buffer of 64 bytes.
     * @param identity The identity to save.
     * @throws Exception
     */
    @Throws(Exception::class)
    fun saveIdentity(sessionId: String, authFactor: AuthFactor, challenge: String, rawTMRSymKey: ByteArray, identity: ByteArray) {
        mobileSSKSTMR.saveIdentity(sessionId, authFactor.toMobileSdk(), challenge, rawTMRSymKey, identity)
    }

    /**
     * Save the Seald account to SSKS.
     *
     * @param sessionId Session ID given by SSKS to your app's server.
     * @param authFactor Authentication method of this user, to which SSKS has sent a challenge at the request of your app's server.
     * @param challenge The challenge sent by SSKS to the user's authentication method. Set to empty string if the SSKS server replies to challenge_send API call with `must_authenticate` set to `false`.
     * @param rawTMRSymKey The raw encryption key used to encrypt / decrypt the stored identity keys. This *MUST* be a cryptographically random buffer of 64 bytes.
     * @param identity The identity to save.
     * @throws Exception
     */
    @Throws(Exception::class)
    suspend fun saveIdentityAsync(sessionId: String, authFactor: AuthFactor, challenge: String, rawTMRSymKey: ByteArray, identity: ByteArray) = withContext(Dispatchers.Default) {
        saveIdentity(sessionId, authFactor, challenge, rawTMRSymKey, identity)
    }

    /**
     * Retrieve the Seald account previously saved with `SealdSSKSTmrPlugin.saveIdentity`.
     *
     * @param sessionId Session ID given by SSKS to your app's server.
     * @param authFactor Authentication method of this user, to which SSKS has sent a challenge at the request of your app's server.
     * @param challenge The challenge sent by SSKS to the user's authentication method.
     * @param rawTMRSymKey The raw encryption key used to encrypt / decrypt the stored identity keys. This *MUST* be a cryptographically random buffer of 64 bytes.
     * @return An [RetrieveIdentityResponse] instance, containing `identity`, the retrieved identity, `shouldRenewKey`, a boolean set to true is the user private key should be renewed (using sealdSDKInstance.renewKeys()), and `authenticatedSessionId`, a new authenticated sessionId, that you can use to perform further SSKS TMR operations without challenge.
     * @throws Exception
     */
    @Throws(Exception::class)
    fun retrieveIdentity(sessionId: String, authFactor: AuthFactor, challenge: String, rawTMRSymKey: ByteArray): RetrieveIdentityResponse {
        val internalResp = mobileSSKSTMR.retrieveIdentity(sessionId, authFactor.toMobileSdk(), challenge, rawTMRSymKey)
        return RetrieveIdentityResponse.fromMobileSdk(internalResp)
    }

    /**
     * Retrieve the Seald account previously saved with `SealdSSKSTmrPlugin.saveIdentity`.
     *
     * @param sessionId Session ID given by SSKS to your app's server.
     * @param authFactor Authentication method of this user, to which SSKS has sent a challenge at the request of your app's server.
     * @param challenge The challenge sent by SSKS to the user's authentication method.
     * @param rawTMRSymKey The raw encryption key used to encrypt / decrypt the stored identity keys. This *MUST* be a cryptographically random buffer of 64 bytes.
     * @return An [RetrieveIdentityResponse] instance, containing `identity`, the retrieved identity, `shouldRenewKey`, a boolean set to true is the user private key should be renewed (using sealdSDKInstance.renewKeys()), and `authenticatedSessionId`, a new authenticated sessionId, that you can use to perform further SSKS TMR operations without challenge.
     * @throws Exception
     */
    @Throws(Exception::class)
    suspend fun retrieveIdentityAsync(sessionId: String, authFactor: AuthFactor, challenge: String, rawTMRSymKey: ByteArray): RetrieveIdentityResponse = withContext(Dispatchers.Default) {
        return@withContext retrieveIdentity(sessionId, authFactor, challenge, rawTMRSymKey)
    }
}