package io.seald.seald_sdk

import java.lang.IllegalArgumentException
import java.time.Duration
import java.time.Instant

// We cannot directly pass an array to GO. We need to instantiate a StringArray struct, and pass it.
// Maybe one day... https://github.com/golang/go/issues/13445
internal fun arrayToStringArray(array: Array<String>): io.seald.seald_sdk_internals.mobile_sdk.StringArray {
    var sa = io.seald.seald_sdk_internals.mobile_sdk.StringArray()
    val arrayIterator = array.iterator()
    while (arrayIterator.hasNext()) {
        sa = sa.add(arrayIterator.next())
    }
    return sa
}

internal fun stringArrayToArray(stringArray: io.seald.seald_sdk_internals.mobile_sdk.StringArray): Array<String> {
    val result = Array(stringArray.size().toInt()) { "" }
    for (i in 0 until stringArray.size().toInt()) {
        result[i] = stringArray.get(i.toLong())
    }
    return result
}

/**
 * ClearFile represents a decrypted file.
 *
 * @property filename The filename of the decrypted file.
 * @property sessionId The ID of the [EncryptionSession] to which this file belongs.
 * @property fileContent The content of the decrypted file.
 */
data class ClearFile(
    val filename: String,
    val sessionId: String,
    val fileContent: ByteArray
)

/**
 * AccountInfo is returned when calling [SealdSDK.createAccount] or [SealdSDK.getCurrentAccountInfo],
 * containing information about the local account.
 *
 * @property userId The ID of the current user for this SDK instance.
 * @property deviceId The ID of the current device for this SDK instance.
 * @property deviceExpires The [Instant] at which the current device keys expire. For continued operation, renew your device keys before this date. `null` if it is not known locally: use [SealdSDK.updateCurrentDevice] to retrieve it.
 */
data class AccountInfo(
    val userId: String,
    val deviceId: String,
    val deviceExpires: Instant?,
) {
    internal companion object {
        internal fun fromMobileSdk(accountInfo: io.seald.seald_sdk_internals.mobile_sdk.AccountInfo?): AccountInfo? {
            if (accountInfo == null) {
                return null
            }
            return AccountInfo(
                             userId = accountInfo.userId,
                             deviceId = accountInfo.deviceId,
                             deviceExpires = if (accountInfo.deviceExpires == 0L) null else Instant.ofEpochSecond(accountInfo.deviceExpires)
                         )
        }
    }
}

/**
 * CreateSubIdentityResponse represents a newly created sub identity.
 *
 * @property deviceId The ID of the newly created device.
 * @property backupKey The identity export of the newly created sub-identity.
 */
data class CreateSubIdentityResponse(
    val deviceId: String,
    val backupKey: ByteArray
)

/**
 * BeardError represents an error returned by the server.
 * It contains a specific `id` and `code` to determine the underlying reason.
 *
 * @property id The error id.
 * @property code The error code.
 */
data class BeardError(
    val id: String,
    val code: String
)

/**
 * ConnectorType represents the allowed values for Connector types:
 * - `EM` for email connectors
 * - `AP` for App connectors
 */
enum class ConnectorType(val type: String) {
    EM("EM"),
    AP("AP");

    internal companion object {
        internal fun fromString(t: String): ConnectorType {
            for (ct in ConnectorType.values()) {
                if (ct.type == t) {
                    return ct
                }
            }
            throw IllegalArgumentException("Invalid ConnectorType value: $t")
        }
    }
}

/**
 * ConnectorState represents the allowed values for Connector states.
 */
enum class ConnectorState(val state: String) {
    PENDING("PE"),
    VALIDATED("VO"),
    REVOKED("RE"),
    REMOVED("RM");

    internal companion object {
        internal fun fromString(s: String): ConnectorState {
            for (cs in ConnectorState.values()) {
                if (cs.state == s) {
                    return cs
                }
            }
            throw IllegalArgumentException("Invalid ConnectorState value: $s")
        }
    }
}

/**
 * ConnectorTypeValue is a simplified representation of a connector for which we don't know all details.
 */
data class ConnectorTypeValue(
    val type: ConnectorType,
    val value: String
) {

    internal companion object {
        internal fun toMobileSdkArray(connectorsArray: Array<ConnectorTypeValue>): io.seald.seald_sdk_internals.mobile_sdk.ConnectorTypeValueArray {
            val result = io.seald.seald_sdk_internals.mobile_sdk.ConnectorTypeValueArray()
            for (c in connectorsArray) {
                result.add(c.toMobileSdk())
            }
            return result
        }
    }

    internal fun toMobileSdk(): io.seald.seald_sdk_internals.mobile_sdk.ConnectorTypeValue {
        val res = io.seald.seald_sdk_internals.mobile_sdk.ConnectorTypeValue()
        res.type = this.type.type
        res.value = this.value
        return res
    }
}

/**
 * Connector represents all details about a connector.
 */
data class Connector(
    val sealdId: String,
    val type: ConnectorType,
    val value: String,
    val id: String,
    val state: ConnectorState
) {
    internal companion object {
        internal fun fromMobileSdk(c: io.seald.seald_sdk_internals.mobile_sdk.Connector): Connector {
            return Connector(
                sealdId = c.sealdId,
                type = ConnectorType.fromString(c.type),
                value = c.value,
                id = c.id,
                state = ConnectorState.fromString(c.state)
            )
        }

        internal fun fromMobileSdkArray(connectorsArray: io.seald.seald_sdk_internals.mobile_sdk.ConnectorsArray): Array<Connector> {
            return Array(
                size = connectorsArray.size().toInt()
            ) { i -> fromMobileSdk(connectorsArray.get(i.toLong())) }
        }

        internal fun toMobileSdkArray(connectorsArray: Array<Connector>): io.seald.seald_sdk_internals.mobile_sdk.ConnectorsArray {
            val result = io.seald.seald_sdk_internals.mobile_sdk.ConnectorsArray()
            for (c in connectorsArray) {
                result.add(c.toMobileSdk())
            }
            return result
        }
    }

    internal fun toMobileSdk(): io.seald.seald_sdk_internals.mobile_sdk.Connector {
        val res = io.seald.seald_sdk_internals.mobile_sdk.Connector()
        res.sealdId = this.sealdId
        res.type = this.type.type
        res.value = this.value
        res.id = this.id
        res.state = this.state.state
        return res
    }
}

/**
 * PreValidationToken represents a way for your server to authorize the adding of a connector.
 */
data class PreValidationToken(
    val domainValidationKeyId: String,
    val nonce: String,
    val token: String
) {
    internal fun toMobileSdk(): io.seald.seald_sdk_internals.mobile_sdk.PreValidationToken {
        val res = io.seald.seald_sdk_internals.mobile_sdk.PreValidationToken()
        res.domainValidationKeyId = this.domainValidationKeyId
        res.nonce = this.nonce
        res.token = this.token
        return res
    }
}

/**
 * Options for [SealdSDK.massReencrypt] function.
 *
 * @property retries Number of times to retry. Defaults to 3.
 * @property retrieveBatchSize Default to 1000.
 * @property waitBetweenRetries Time to wait between retries. Defaults to 3 seconds.
 * @property waitProvisioning Whether to wait for provisioning (new behaviour) or not. Defaults to true.
 * @property waitProvisioningTime Time to wait if device is not provisioned on the server yet. The actual wait time will be increased on subsequent tries, by `waitProvisioningTimeStep`, up to `waitProvisioningTimeMax`. Defaults to 5 seconds.
 * @property waitProvisioningTimeMax Maximum time to wait if device is not provisioned on the server yet. Defaults to 10 seconds.
 * @property waitProvisioningTimeStep Amount to increase the time to wait if device is not provisioned on the server yet. Defaults to 1 second.
 * @property waitProvisioningRetries Maximum number of tries to check if the device is provisioned yet. Defaults to 100.
 * @property forceLocalAccountUpdate Whether to update the local account before trying the reencryption.
 */
data class MassReencryptOptions @JvmOverloads constructor(
    var retries: Int = 3,
    var retrieveBatchSize: Int = 1000,
    var waitBetweenRetries: Duration = Duration.ofSeconds(3),
    var waitProvisioning: Boolean = true,
    var waitProvisioningTime: Duration = Duration.ofSeconds(5),
    var waitProvisioningTimeMax: Duration = Duration.ofSeconds(10),
    var waitProvisioningTimeStep: Duration = Duration.ofSeconds(1),
    var waitProvisioningRetries: Int = 100,
    var forceLocalAccountUpdate: Boolean = false
) {
    internal fun toMobileSdk(): io.seald.seald_sdk_internals.mobile_sdk.MassReencryptOptions {
        val res = io.seald.seald_sdk_internals.mobile_sdk.MassReencryptOptions()
        res.retries = retries.toLong()
        res.retrieveBatchSize = retrieveBatchSize.toLong()
        res.waitBetweenRetries = waitBetweenRetries.toMillis()
        res.waitProvisioning = waitProvisioning
        res.waitProvisioningTime = waitProvisioningTime.toMillis()
        res.waitProvisioningTimeMax = waitProvisioningTimeMax.toMillis()
        res.waitProvisioningTimeStep = waitProvisioningTimeStep.toMillis()
        res.waitProvisioningRetries = waitProvisioningRetries.toLong()
        res.forceLocalAccountUpdate = forceLocalAccountUpdate
        return res
    }
}

/**
 * Represents the results of a call to [SealdSDK.massReencrypt].
 *
 * @property reencrypted The number of session keys that were reencrypted for the given device.
 * @property failed The number of session keys that could not be reencrypted for the given device.
 */
data class MassReencryptResponse(
    val reencrypted: Int,
    val failed: Int
) {
    internal companion object {
        internal fun fromMobileSdk(d: io.seald.seald_sdk_internals.mobile_sdk.MassReencryptResponse): MassReencryptResponse {
            return MassReencryptResponse(
                reencrypted = d.reencrypted.toInt(),
                failed = d.failed.toInt()
            )
        }
    }
}

/**
 * Represents a device of the current account which is missing some keys, and for which you probably want to call [SealdSDK.massReencrypt].
 *
 * @property deviceId The ID of the device which is missing some keys.
 * @property count The number of keys missing for this device. Warning: the count may be lower than reality if the device was recently created, as the server may still be provisioning keys for this device. In that case, you can still call [SealdSDK.massReencrypt] for that device.
 */
data class DeviceMissingKeys(
    val deviceId: String,
    val count: Int
) {
    internal companion object {
        internal fun fromMobileSdk(d: io.seald.seald_sdk_internals.mobile_sdk.DeviceMissingKeys): DeviceMissingKeys {
            return DeviceMissingKeys(
                deviceId = d.deviceId,
                count = d.count.toInt()
            )
        }

        internal fun fromMobileSdkArray(array: io.seald.seald_sdk_internals.mobile_sdk.DevicesMissingKeysArray): Array<DeviceMissingKeys> {
            return Array(
                size = array.size().toInt()
            ) { i -> fromMobileSdk(array.get(i.toLong())) }
        }
    }
}

/**
 * Represents the status of an operation on single user/device.
 *
 * @property success The status of the action: true if succeeded, false otherwise.
 * @property errorCode An error message, or an empty string.
 */
data class ActionStatus(
    val success: Boolean,
    val errorCode: String
) {
    internal companion object {

        internal fun fromMobileSdkArray(goArray: io.seald.seald_sdk_internals.mobile_sdk.ActionStatusArray): Map<String, ActionStatus> {
            val res = mutableMapOf<String, ActionStatus>()
            for (i in 0..(goArray.size().toInt() -1)) {
                val goEl = goArray.get(i.toLong())
                var actionStatus = ActionStatus(success = goEl.success, errorCode = goEl.errorCode)
                res[goEl.id] = actionStatus
            }
            return res
        }
    }
}
