package io.scaleplan.templating;

import org.jetbrains.annotations.NotNull;

import java.util.Objects;
import java.util.regex.Pattern;

public class TemplateOptions {
    public static final Pattern DEFAULT_PATTERN =
            Pattern.compile("(\\{([A-Za-z0-9_:.-]*)(?:(?:\\|)([^}]*))?})");

    static class Builder {
        private Pattern pattern = null;
        private char lastDelimiter = '/';
        private String restKey = "rest";

        public Builder setPattern(@NotNull Pattern pattern) {
            this.pattern = pattern;
            return this;
        }

        public Builder setLastDelimiter(char lastDelimiter) {
            this.lastDelimiter = lastDelimiter;
            return this;
        }

        public Builder setRestKey(String restKey) {
            this.restKey = restKey;
            return this;
        }

        public TemplateOptions build() {
            if (this.pattern == null) {
                this.pattern = DEFAULT_PATTERN;
            }
            return new TemplateOptions(this);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public Pattern getPattern() {
        return pattern;
    }

    public char getLastDelimiter() {
        return lastDelimiter;
    }

    public String getRestKey() {
        return restKey;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        TemplateOptions that = (TemplateOptions) o;
        return lastDelimiter == that.lastDelimiter && pattern.equals(that.pattern) && restKey.equals(that.restKey);
    }

    @Override
    public int hashCode() {
        return Objects.hash(pattern, lastDelimiter, restKey);
    }

    private TemplateOptions(Builder builder) {
        this.pattern = builder.pattern;
        this.lastDelimiter = builder.lastDelimiter;
        this.restKey = builder.restKey;
    }

    private final Pattern pattern;
    private final char lastDelimiter;
    private final String restKey;
}
