package io.scaleplan.templating;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Template implements Iterable<TemplateFragment> {
    private final static TemplateOptions DEFAULT_OPTIONS =
            TemplateOptions.builder().build();

    private final List<TemplateFragment> fragments;
    private final String materializedText;
    private final TemplateOptions options;

    private Template(@NotNull final List<TemplateFragment> fragments) {
        this(fragments, DEFAULT_OPTIONS);
    }

    private Template(
            @NotNull final List<TemplateFragment> fragments,
            @NotNull final TemplateOptions options
    ) {
        List<TemplateFragment> optimizedFrags = optimize(Objects.requireNonNull(fragments));
        // the template is materialized if it has zero or 1 text fragments and no placeholders
        if (optimizedFrags.isEmpty()) {
            materializedText = "";
        } else if (optimizedFrags.size() == 1 && optimizedFrags.get(0).isText()) {
            materializedText = optimizedFrags.get(0).getValue();
        } else {
            materializedText = null;
        }
        this.fragments = optimizedFrags;
        this.options = options;
    }

    public static Template fromFragments(@Nullable List<TemplateFragment> fragments) {
        if (fragments == null) return new Template(Collections.emptyList());
        else return new Template(Collections.unmodifiableList(fragments));
    }

    public static Template compile(@NotNull final String text) {
        return compile(text, null);
    }

    public static Template compile(
            @NotNull final String text,
            @Nullable final Map<String, Object> context
    ) {
        return compile(text, context, DEFAULT_OPTIONS);
    }

    public static Template compile(
            @NotNull final String text,
            @Nullable final Map<String, Object> context,
            @NotNull final TemplateOptions options
    ) {
        return compile(
                text,
                options.getPattern().matcher(text),
                options.getRestKey(),
                context
        );
    }

    private static Template compile(
            @NotNull final String text,
            @NotNull final Matcher matcher,
            @NotNull final String restKey,
            @Nullable final Map<String, Object> context
    ) {
        Objects.requireNonNull(text, "text is required for Template#compile");
        Objects.requireNonNull(matcher, "matcher is required for Template#compile");
        List<TemplateFragment> fragments = new ArrayList<>();
        int prevEnd = 0;
        int placeholderCounter = 0;
        while (matcher.find()) {
            int start = matcher.start();
            boolean isEscape = false;
            // check whether the placeholder is escaped
            if (start > 0 && text.charAt(start - 1) == '\\') {
                // This is an escape, skip
                isEscape = true;
                start--;
            }
            // check and add regular text
            if (start != prevEnd) {
                fragments.add(TemplateFragment.create(text.subSequence(prevEnd, start).toString()));
            }
            // check and add placeholder
            if (isEscape) {
                fragments.add(TemplateFragment.create(matcher.group(1)));
            } else {
                // materialized template
                // check whether there is a context and the placeholder is in it
                placeholderCounter++;
                String key = matcher.group(2);
                if (key.equals(restKey)) {
                    // rest placeholder
                    if (matcher.end() != text.length()) {
                        throw new InvalidTemplateException("Rest placeholder can only appear at the end of a template.");
                    } else {
                        fragments.add(TemplateFragment.create(matcher.group(3), restKey));
                    }
                } else {
                    if (key.isEmpty()) {
                        key = Integer.toString(placeholderCounter);
                    }
                    if (context != null && context.containsKey(key)) {
                        fragments.add(TemplateFragment.create((String) context.get(key)));
                    } else {
                        fragments.add(TemplateFragment.create(matcher.group(3), key));
                    }
                }
            }
            prevEnd = matcher.end();
        }
        if (prevEnd < text.length()) {
            fragments.add(TemplateFragment.create(text.subSequence(prevEnd, text.length()).toString()));
        }
        return new Template(fragments);
    }

    @NotNull
    @Override
    public Iterator<TemplateFragment> iterator() {
        return fragments.iterator();
    }

    @Override
    public Spliterator<TemplateFragment> spliterator() {
        return fragments.spliterator();
    }

    public int size() {
        return fragments.size();
    }

    public boolean isEmpty() {
        return fragments.isEmpty();
    }

    /**
     * Returns true if the this template has no placeholders
     *
     * @return
     */
    public boolean isMaterialized() {
        return materializedText != null;
    }

    public String getMaterializedText() {
        return materializedText;
    }

    public TemplateOptions getOptions() {
        return options;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder(fragments.size());
        for (TemplateFragment frag : fragments) {
            builder.append(frag.toString());
        }
        return builder.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        Template template = (Template) o;
        return fragments.equals(template.fragments) && options.equals(template.options);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fragments, options);
    }

    /*
    public String render(
            @Nullable final List<RendererSpec> namespaceSpecs,
            @Nullable final Function<String, String> fallbackRenderer
    ) {
        return render(namespaceSpecs, fallbackRenderer, null);
    }

    public String renderStrict(
            @Nullable final List<RendererSpec> namespaceSpecs,
            @Nullable final Function<String, String> fallbackRenderer
    ) {
        return renderStrict(namespaceSpecs, fallbackRenderer, null);
    }

    public String render(
            @Nullable final List<RendererSpec> namespaceSpecs,
            @Nullable final Function<String, String> fallbackRenderer,
            @Nullable final Function<String, String> placeholderValueTransformer
    ) {
        return render(namespaceSpecs, fallbackRenderer, placeholderValueTransformer, false);
    }

    public String renderStrict(
            @Nullable final List<RendererSpec> namespaceSpecs,
            @Nullable final Function<String, String> fallbackRenderer,
            @Nullable final Function<String, String> placeholderValueTransformer
    ) {
        return render(namespaceSpecs, fallbackRenderer, placeholderValueTransformer, true);
    }
     */

    /*
    public String render(
            @Nullable final List<RendererSpec> namespaceSpecs,
            @Nullable final Function<String, String> fallbackRenderer,
            @Nullable final Function<String, String> placeholderValueTransformer,
            boolean strict
    ) {
    }
     */

    public String render(
            @NotNull Function<String, String> keyToValue,
            boolean strict) {

        StringBuilder builder = new StringBuilder();
        for (TemplateFragment frag : fragments) {
            String key = frag.getKey();
            String value;
            if (key == null) {
                value = frag.getValue();
            } else {
                value = keyToValue.apply(key);
                if (value == null) {
                    if (frag.hasDefault()) {
                        value = frag.getValue();
                    } else if (strict) {
                        throw new MissingValueException(key);
                    } else {
                        value = "";
                    }
                }
            }
            builder.append(value);
        }
        return builder.toString();
    }

    public Map<String, String> extract(@NotNull final String text) {
        Objects.requireNonNull(text);
        // TODO: return rest text in placeholder: *
        // first check if this is a materialized template
        if (isMaterialized()) {
            // Yes, it is.
            String mt = materializedText;
            if (mt.regionMatches(0, text, 0, mt.length())) {
                return Collections.emptyMap();
            } else {
                return null;
            }
        }
        Map<String, String> result = new HashMap<>((size() / 2) + 1);
        int currentOffset = 0;
        int lastFragmentIndex = fragments.size() - 1;
        String restKey = options.getRestKey();
        for (int i = 0; i < fragments.size(); i++) {
            TemplateFragment frag = fragments.get(i);
            if (frag.isText()) {
                String fragValue = frag.getValue();
                int regionLength = fragValue.length();
                if (!text.regionMatches(currentOffset, fragValue, 0, regionLength)) {
                    return null;
                }
                currentOffset += regionLength;
                continue;
            }
            // check whether this is a rest placeholder
            if (frag.getKey().equals(restKey)) {
                result.put(frag.getKey(), text.substring(currentOffset));
                return result;
            }
            // if this is the last fragment, nextTestPos is the index of the given last delimiter
            int nextTextPos = 0;
            if (i == lastFragmentIndex) {
                // if last delimiter is found then there's no match
                nextTextPos = text.indexOf(options.getLastDelimiter(), currentOffset);
                if (nextTextPos > 0) return null;
                else nextTextPos = text.length();
            } else {
                // try to find the next text pos
                // check whether the next fragment is a text fragment
                TemplateFragment nextFragment = fragments.get(i + 1);
                if (nextFragment.isText()) {
                    // empty text is optimized away, so len(text) is always >= 1
                    nextTextPos = text.indexOf(nextFragment.getValue().charAt(0), currentOffset);
                } else {
                    // next is a placeholder.
                    // this placeholder has no chance to match
                    if (frag.hasDefault()) {
                        result.put(frag.getKey(), frag.getValue());
                    } else {
                        // no match
                        return null;
                    }
                }
            }
            if (nextTextPos < 0) {
                // no chance to match
                return null;
            }
            if (currentOffset == nextTextPos) {
                // no match
                // if there's a default, use it.
                if (!frag.hasDefault()) {
                    // no defaults, report no match
                    return null;
                }
                result.put(frag.getKey(), frag.getValue());
                continue;
            }
            String subText = text.substring(currentOffset, nextTextPos);
            result.put(frag.getKey(), subText);
            currentOffset = nextTextPos;
        }
        // if there are non-extracted parts of the text, then there is no match
        if (currentOffset < text.length()) return null;
        return result;
    }

    public String toRegexString() {
        if (isMaterialized()) {
            return String.format("^%s$", Pattern.quote(materializedText));
        }
        StringBuilder sb = new StringBuilder();
        sb.append("^");
        int lastFragmentIndex = fragments.size() - 1;
        String restKey = options.getRestKey();
        for (int i = 0; i < fragments.size(); i++) {
            TemplateFragment frag = fragments.get(i);
            if (frag.isText()) {
                sb.append(Pattern.quote(frag.getValue()));
                continue;
            }
            // check whether this is a rest placeholder
            if (frag.getKey().equals(restKey)) {
                sb.append(String.format("(?<%s>.*)", restKey));
                break;
            }

            sb.append("(?<");
            sb.append(frag.getKey());
            sb.append('>');

            if (i == lastFragmentIndex) {
                if (options.getLastDelimiter() == ']') {
                    sb.append("[^\\]]+");
                } else {
                    sb.append(String.format("[^%c]+", options.getLastDelimiter()));
                }
            } else {
                TemplateFragment nextFragment = fragments.get(i + 1);
                // check whether the next fragment is text
                if (nextFragment.isText()) {
                    // empty text is optimized away, so len(text) is always >= 1
                    char nextChar = nextFragment.getValue().charAt(0);
                    sb.append("[^");
                    if (nextChar == ']') sb.append("\\]");
                    else sb.append(nextChar);
                    sb.append("]+");
                } else {
                    // next is a placeholder.
                    // this placeholder has no chance to match
                    sb.append("()");
                }
            }
            sb.append(frag.hasDefault() ? ")?" : ')');
        }
        sb.append("$");
        return sb.toString();
    }

    /**
     * Merges consecutive text blocks
     */
    static List<TemplateFragment> optimize(final List<TemplateFragment> fragments) {
        StringBuilder stringBuilder = new StringBuilder();
        List<TemplateFragment> newFragments = new ArrayList<>(fragments.size());
        for (TemplateFragment frag : fragments) {
            if (frag.isText()) {
                stringBuilder.append(frag.getValue());
            } else {
                if (stringBuilder.length() > 0) {
                    newFragments.add(TemplateFragment.create(stringBuilder.toString()));
                    stringBuilder.setLength(0);
                }
                newFragments.add(frag);
            }
        }
        if (stringBuilder.length() > 0) {
            newFragments.add(TemplateFragment.create(stringBuilder.toString()));
        }
        return newFragments;
    }
}
