package io.quarkiverse.reactive.messsaging.nats.jetstream;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code quarkus-jetstream} connector.
*/
 public class JetStreamConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new JetStreamConnectorCommonConfiguration.
   */
  public JetStreamConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the stream value from the configuration.
  * Attribute Name: stream
  * Description: The stream to subscribe or publish messages to
  * @return the stream
  */
  public Optional<String> getStream() {
    return config.getOptionalValue("stream", String.class);
  }

  /**
  * Gets the subject value from the configuration.
  * Attribute Name: subject
  * Description: The subject to subscribe or publish messages to
  * @return the subject
  */
  public Optional<String> getSubject() {
    return config.getOptionalValue("subject", String.class);
  }

  /**
  * Gets the trace-enabled value from the configuration.
  * Attribute Name: trace-enabled
  * Description: Enable traces for publisher or subscriber
  * Default Value: true
  * @return the trace-enabled
  */
  public Boolean getTraceEnabled() {
    return config.getOptionalValue("trace-enabled", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the auto-configure value from the configuration.
  * Attribute Name: auto-configure
  * Description: Auto configure subject on NATS
  * Default Value: true
  * @return the auto-configure
  */
  public Boolean getAutoConfigure() {
    return config.getOptionalValue("auto-configure", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  public void validate() {
  }
}
