package io.quarkiverse.hivemqclient.smallrye.reactive;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;
import org.eclipse.microprofile.reactive.messaging.spi.ConnectorFactory;

/**
 * Extracts the common configuration for the {@code smallrye-mqtt-hivemq} connector.
*/
 public class HiveMQMqttConnectorCommonConfiguration {
  protected final Config config;

  /**
   * Creates a new HiveMQMqttConnectorCommonConfiguration.
   */
  public HiveMQMqttConnectorCommonConfiguration(Config config) {
    this.config = config;
  }

  /**
   * @return the connector configuration
   */
  public Config config() {
    return this.config;
  }

  /**
   * Retrieves the value stored for the given alias.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> Optional<T> getFromAlias(String alias, Class<T> type) {
    return ConfigProvider.getConfig().getOptionalValue(alias, type);
  }

  /**
   * Retrieves the value stored for the given alias. Returns the default value if not present.
   * @param alias the attribute alias, must not be {@code null} or blank
   * @param type the targeted type
   * @param defaultValue the default value
   * @param <T> the targeted type
   * @return the configuration value for the given alias, empty if not set
   */
  protected <T> T getFromAliasWithDefaultValue(String alias, Class<T> type, T defaultValue) {
    return getFromAlias(alias, type).orElse(defaultValue);
  }

  /**
   * @return the channel name
   */
  public String getChannel() {
    return config.getValue(ConnectorFactory.CHANNEL_NAME_ATTRIBUTE, String.class);
  }

  /**
  * Gets the client-id value from the configuration.
  * Attribute Name: client-id
  * Description: Set the client identifier
  * @return the client-id
  */
  public Optional<String> getClientId() {
    return config.getOptionalValue("client-id", String.class);
  }

  /**
  * Gets the auto-generated-client-id value from the configuration.
  * Attribute Name: auto-generated-client-id
  * Description: Set if the MQTT client must generate clientId automatically
  * Default Value: true
  * @return the auto-generated-client-id
  */
  public Boolean getAutoGeneratedClientId() {
    return config.getOptionalValue("auto-generated-client-id", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the auto-keep-alive value from the configuration.
  * Attribute Name: auto-keep-alive
  * Description: Set if the MQTT client must handle `PINGREQ` automatically
  * Default Value: true
  * @return the auto-keep-alive
  */
  public Boolean getAutoKeepAlive() {
    return config.getOptionalValue("auto-keep-alive", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the ssl value from the configuration.
  * Attribute Name: ssl
  * Description: Set whether SSL/TLS is enabled
  * Default Value: false
  * @return the ssl
  */
  public Boolean getSsl() {
    return config.getOptionalValue("ssl", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the keep-alive-seconds value from the configuration.
  * Attribute Name: keep-alive-seconds
  * Description: Set the keep alive timeout in seconds
  * Default Value: 30
  * @return the keep-alive-seconds
  */
  public Integer getKeepAliveSeconds() {
    return config.getOptionalValue("keep-alive-seconds", Integer.class)
     .orElse(Integer.valueOf("30"));
  }

  /**
  * Gets the max-inflight-queue value from the configuration.
  * Attribute Name: max-inflight-queue
  * Description: Set max count of unacknowledged messages
  * Default Value: 10
  * @return the max-inflight-queue
  */
  public Integer getMaxInflightQueue() {
    return config.getOptionalValue("max-inflight-queue", Integer.class)
     .orElse(Integer.valueOf("10"));
  }

  /**
  * Gets the auto-clean-session value from the configuration.
  * Attribute Name: auto-clean-session
  * Description: Set to start with a clean session (`true` by default)
  * Default Value: true
  * @return the auto-clean-session
  */
  public Boolean getAutoCleanSession() {
    return config.getOptionalValue("auto-clean-session", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the will-flag value from the configuration.
  * Attribute Name: will-flag
  * Description: Set if will information are provided on connection
  * Default Value: false
  * @return the will-flag
  */
  public Boolean getWillFlag() {
    return config.getOptionalValue("will-flag", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the will-retain value from the configuration.
  * Attribute Name: will-retain
  * Description: Set if the will message must be retained
  * Default Value: false
  * @return the will-retain
  */
  public Boolean getWillRetain() {
    return config.getOptionalValue("will-retain", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the will-qos value from the configuration.
  * Attribute Name: will-qos
  * Description: Set the QoS level for the will message
  * Default Value: 0
  * @return the will-qos
  */
  public Integer getWillQos() {
    return config.getOptionalValue("will-qos", Integer.class)
     .orElse(Integer.valueOf("0"));
  }

  /**
  * Gets the max-message-size value from the configuration.
  * Attribute Name: max-message-size
  * Description: Set max MQTT message size in bytes
  * Default Value: 8092
  * @return the max-message-size
  */
  public Integer getMaxMessageSize() {
    return config.getOptionalValue("max-message-size", Integer.class)
     .orElse(Integer.valueOf("8092"));
  }

  /**
  * Gets the reconnect-attempts value from the configuration.
  * Attribute Name: reconnect-attempts
  * Description: Set the max reconnect attempts
  * Default Value: 5
  * @return the reconnect-attempts
  */
  public Integer getReconnectAttempts() {
    return config.getOptionalValue("reconnect-attempts", Integer.class)
     .orElse(Integer.valueOf("5"));
  }

  /**
  * Gets the reconnect-interval-seconds value from the configuration.
  * Attribute Name: reconnect-interval-seconds
  * Description: Set the reconnect interval in seconds
  * Default Value: 1
  * @return the reconnect-interval-seconds
  */
  public Integer getReconnectIntervalSeconds() {
    return config.getOptionalValue("reconnect-interval-seconds", Integer.class)
     .orElse(Integer.valueOf("1"));
  }

  /**
  * Gets the username value from the configuration.
  * Attribute Name: username
  * Description: Set the username to connect to the server
  * @return the username
  */
  public Optional<String> getUsername() {
    return config.getOptionalValue("username", String.class);
  }

  /**
  * Gets the password value from the configuration.
  * Attribute Name: password
  * Description: Set the password to connect to the server
  * @return the password
  */
  public Optional<String> getPassword() {
    return config.getOptionalValue("password", String.class);
  }

  /**
  * Gets the connect-timeout-seconds value from the configuration.
  * Attribute Name: connect-timeout-seconds
  * Description: Set the connect timeout (in seconds)
  * Default Value: 60
  * @return the connect-timeout-seconds
  */
  public Integer getConnectTimeoutSeconds() {
    return config.getOptionalValue("connect-timeout-seconds", Integer.class)
     .orElse(Integer.valueOf("60"));
  }

  /**
  * Gets the trust-all value from the configuration.
  * Attribute Name: trust-all
  * Description: Set whether all server certificates should be trusted
  * Default Value: false
  * @return the trust-all
  */
  public Boolean getTrustAll() {
    return config.getOptionalValue("trust-all", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the host value from the configuration.
  * Attribute Name: host
  * Description: Set the MQTT server host name/IP
  * Mandatory: yes
  * @return the host
  */
  public String getHost() {
    return config.getOptionalValue("host", String.class)
        .orElseThrow(() -> new IllegalArgumentException("The attribute `host` on connector 'smallrye-mqtt-hivemq' (channel: " + getChannel() + ") must be set"));
  }

  /**
  * Gets the port value from the configuration.
  * Attribute Name: port
  * Description: Set the MQTT server port. Default to 8883 if ssl is enabled, or 1883 without ssl
  * @return the port
  */
  public Optional<Integer> getPort() {
    return config.getOptionalValue("port", Integer.class);
  }

  /**
  * Gets the server-name value from the configuration.
  * Attribute Name: server-name
  * Description: Set the SNI server name
  * @return the server-name
  */
  public Optional<String> getServerName() {
    return config.getOptionalValue("server-name", String.class);
  }

  /**
  * Gets the topic value from the configuration.
  * Attribute Name: topic
  * Description: Set the MQTT topic. If not set, the channel name is used
  * @return the topic
  */
  public Optional<String> getTopic() {
    return config.getOptionalValue("topic", String.class);
  }

  /**
  * Gets the qos value from the configuration.
  * Attribute Name: qos
  * Description: Set the QoS level when subscribing to the topic or when sending a message
  * Default Value: 0
  * @return the qos
  */
  public Integer getQos() {
    return config.getOptionalValue("qos", Integer.class)
     .orElse(Integer.valueOf("0"));
  }

  /**
  * Gets the ssl.truststore.type value from the configuration.
  * Attribute Name: ssl.truststore.type
  * Description: Set the truststore type pkcs12, jks, pem]
  * Default Value: jks
  * @return the ssl.truststore.type
  */
  public String getSslTruststoreType() {
    return config.getOptionalValue("ssl.truststore.type", String.class)
     .orElse("jks");
  }

  /**
  * Gets the ssl.truststore.location value from the configuration.
  * Attribute Name: ssl.truststore.location
  * Description: Set the truststore location. In case of pem type this is the cert path
  * @return the ssl.truststore.location
  */
  public Optional<String> getSslTruststoreLocation() {
    return config.getOptionalValue("ssl.truststore.location", String.class);
  }

  /**
  * Gets the ssl.truststore.password value from the configuration.
  * Attribute Name: ssl.truststore.password
  * Description: Set the truststore password. In case of pem type this is not necessary
  * @return the ssl.truststore.password
  */
  public Optional<String> getSslTruststorePassword() {
    return config.getOptionalValue("ssl.truststore.password", String.class);
  }

  /**
  * Gets the ssl.keystore.type value from the configuration.
  * Attribute Name: ssl.keystore.type
  * Description: Set the keystore type [pkcs12, jks, pem]
  * Default Value: jks
  * @return the ssl.keystore.type
  */
  public String getSslKeystoreType() {
    return config.getOptionalValue("ssl.keystore.type", String.class)
     .orElse("jks");
  }

  /**
  * Gets the ssl.keystore.location value from the configuration.
  * Attribute Name: ssl.keystore.location
  * Description: Set the keystore location. In case of pem type this is the cert path
  * @return the ssl.keystore.location
  */
  public Optional<String> getSslKeystoreLocation() {
    return config.getOptionalValue("ssl.keystore.location", String.class);
  }

  /**
  * Gets the ssl.keystore.password value from the configuration.
  * Attribute Name: ssl.keystore.password
  * Description: Set the keystore password. In case of pem type this is the key path
  * @return the ssl.keystore.password
  */
  public Optional<String> getSslKeystorePassword() {
    return config.getOptionalValue("ssl.keystore.password", String.class);
  }

  /**
  * Gets the ssl.hostVerifier value from the configuration.
  * Attribute Name: ssl.hostVerifier
  * Description: Enable or disable SSL host verification
  * Default Value: true
  * @return the ssl.hostVerifier
  */
  public Boolean getSslHostVerifier() {
    return config.getOptionalValue("ssl.hostVerifier", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the check-topic-enabled value from the configuration.
  * Attribute Name: check-topic-enabled
  * Description: Enable check for liveness/readiness
  * Default Value: false
  * @return the check-topic-enabled
  */
  public Boolean getCheckTopicEnabled() {
    return config.getOptionalValue("check-topic-enabled", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the check-topic-name value from the configuration.
  * Attribute Name: check-topic-name
  * Description: Topic Used to check liveness/readiness
  * Default Value: $SYS/broker/uptime
  * @return the check-topic-name
  */
  public String getCheckTopicName() {
    return config.getOptionalValue("check-topic-name", String.class)
     .orElse("$SYS/broker/uptime");
  }

  /**
  * Gets the readiness-timeout value from the configuration.
  * Attribute Name: readiness-timeout
  * Description: Timeout to declare the MQTT Client not ready
  * Default Value: 20000
  * @return the readiness-timeout
  */
  public Integer getReadinessTimeout() {
    return config.getOptionalValue("readiness-timeout", Integer.class)
     .orElse(Integer.valueOf("20000"));
  }

  /**
  * Gets the liveness-timeout value from the configuration.
  * Attribute Name: liveness-timeout
  * Description: Timeout to declare the MQTT Client not alive
  * Default Value: 120000
  * @return the liveness-timeout
  */
  public Integer getLivenessTimeout() {
    return config.getOptionalValue("liveness-timeout", Integer.class)
     .orElse(Integer.valueOf("120000"));
  }

  public void validate() {
    getHost();
  }
}
