package io.quarkiverse.hivemqclient.smallrye.reactive;

import static io.smallrye.reactive.messaging.mqtt.i18n.MqttLogging.log;
import static java.lang.String.format;

import java.io.File;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import com.hivemq.client.mqtt.MqttClientSslConfigBuilder;
import com.hivemq.client.mqtt.mqtt3.Mqtt3BlockingClient;
import com.hivemq.client.mqtt.mqtt3.Mqtt3Client;
import com.hivemq.client.mqtt.mqtt3.Mqtt3ClientBuilder;
import com.hivemq.client.mqtt.mqtt3.Mqtt3RxClient;
import com.hivemq.client.mqtt.mqtt3.message.connect.connack.Mqtt3ConnAck;

import io.quarkiverse.hivemqclient.ssl.IgnoreHostnameVerifier;
import io.quarkiverse.hivemqclient.ssl.KeyStoreUtil;
import io.smallrye.mutiny.Uni;
import io.smallrye.mutiny.operators.multi.processors.BroadcastProcessor;
import io.smallrye.reactive.messaging.health.HealthReport;
import io.vertx.mutiny.mqtt.messages.MqttPublishMessage;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.TrustManagerFactory;

public class HiveMQClients {

    private static final Map<String, ClientHolder> clients = new ConcurrentHashMap<>();

    private HiveMQClients() {
        // avoid direct instantiation.
    }

    static Uni<Mqtt3RxClient> getConnectedClient(HiveMQMqttConnectorCommonConfiguration options) {

        ClientHolder holder = getHolder(options);

        return holder.connect();
    }

    static ClientHolder getHolder(HiveMQMqttConnectorCommonConfiguration options) {

        String host = options.getHost();
        int def = options.getSsl() ? 8883 : 1883;
        int port = options.getPort().orElse(def);
        String server = options.getServerName().orElse("");
        String clientId = options.getClientId().orElse("");

        String id = host + ":" + port + "<" + server + ">-[" + clientId + "]";
        return clients.computeIfAbsent(id, key -> new ClientHolder(options));
    }

    static Mqtt3RxClient create(HiveMQMqttConnectorCommonConfiguration options) {

        final Mqtt3ClientBuilder builder = Mqtt3Client.builder()
                .serverHost(options.getHost())
                .serverPort(options.getPort().orElse(options.getSsl() ? 8883 : 1883));

        if (options.getAutoGeneratedClientId()) {
            builder.identifier(UUID.randomUUID().toString());
        }
        options.getClientId().ifPresent(clientid -> builder.identifier(clientid));

        options.getUsername().ifPresent(username -> setupBasicAuth(options, username, builder));
        if (options.getSsl()) {
            setupSslConfig(options, builder);
        }

        return builder
                .automaticReconnectWithDefaultConfig()
                .addConnectedListener(context -> {
                    log.info(format("connected to %s:%d", context.getClientConfig().getServerHost(),
                            context.getClientConfig().getServerPort()));
                }).buildRx();
    }

    private static void setupBasicAuth(HiveMQMqttConnectorCommonConfiguration options, String username,
            Mqtt3ClientBuilder builder) {
        builder.simpleAuth()
                .username(username)
                .password(options.getPassword().orElseThrow(
                        () -> new IllegalArgumentException("password null with authentication enabled (username not null)"))
                        .getBytes())
                .applySimpleAuth();
    }

    private static void setupSslConfig(HiveMQMqttConnectorCommonConfiguration options, Mqtt3ClientBuilder builder) {
        final MqttClientSslConfigBuilder.Nested<? extends Mqtt3ClientBuilder> nested = builder.sslConfig();

        String truststoreLocation = options.getSslTruststoreLocation()
                .orElseThrow(() -> new RuntimeException("Missing required 'ssl.truststore.location' property"));
        String truststorePassword = options.getSslTruststorePassword()
                .orElseThrow(() -> new RuntimeException("Missing required 'ssl.truststore.password' property"));
        final TrustManagerFactory trustManagerFactory = KeyStoreUtil.trustManagerFromKeystore(new File(truststoreLocation),
                truststorePassword, options.getSslTruststoreType());

        nested.trustManagerFactory(trustManagerFactory);

        // you must provide a keystore if you are running mTls
        setupMtlsConfig(options, nested);

        if (!options.getSslHostVerifier()) {
            nested.hostnameVerifier(new IgnoreHostnameVerifier());
        }

        nested.applySslConfig();
    }

    private static void setupMtlsConfig(HiveMQMqttConnectorCommonConfiguration options,
            MqttClientSslConfigBuilder.Nested<? extends Mqtt3ClientBuilder> nested) {
        if (options.getSslKeystoreLocation().isPresent() || options.getSslKeystorePassword().isPresent()) {
            String keystoreLocation = options.getSslKeystoreLocation()
                    .orElseThrow(() -> new RuntimeException("Missing required 'ssl.keystore.location' property"));
            String keystorePassword = options.getSslKeystorePassword()
                    .orElseThrow(() -> new RuntimeException("Missing required 'ssl.keystore.password' property"));
            final KeyManagerFactory keyManagerFactory = KeyStoreUtil.keyManagerFromKeystore(new File(keystoreLocation),
                    keystorePassword, keystorePassword, options.getSslKeystoreType());

            nested.keyManagerFactory(keyManagerFactory);
        }
    }

    /**
     * Removed all the stored clients.
     */
    public static void clear() {
        clients.forEach((name, holder) -> holder.close());
        clients.clear();
    }

    public static void checkLiveness(HealthReport.HealthReportBuilder builder) {
        clients.forEach((name, holder) -> builder.add(name, holder.checkLiveness()));
    }

    public static void checkReadiness(HealthReport.HealthReportBuilder builder) {
        clients.forEach((name, holder) -> builder.add(name, holder.checkReadiness()));
    }

    public static class ClientHolder {

        private final Mqtt3RxClient client;
        private final Uni<Mqtt3ConnAck> connection;
        private final int livenessTimeout;
        private final int readinessTimeout;
        private final Boolean checkTopicEnabled;

        private long lastMqttUpdate = 0;
        private final BroadcastProcessor<MqttPublishMessage> messages;

        public ClientHolder(HiveMQMqttConnectorCommonConfiguration options) {
            client = create(options);

            messages = BroadcastProcessor.create();
            livenessTimeout = options.getLivenessTimeout();
            readinessTimeout = options.getReadinessTimeout();
            checkTopicEnabled = options.getCheckTopicEnabled();

            if (checkTopicEnabled) {
                client.toAsync().subscribeWith()
                        .topicFilter(options.getCheckTopicName())
                        .callback(m -> {
                            log.debug(new String(m.getPayloadAsBytes()));
                            lastMqttUpdate = System.currentTimeMillis();
                        })
                        .send();
            }

            connection = Uni.createFrom().future(client.connect().toFuture());
            connection.subscribe().with(c -> log.info(c.getReturnCode()));

        }

        public Uni<Mqtt3RxClient> connect() {
            return connection
                    .map(ignored -> client);
        }

        public boolean checkLiveness() {
            if (!checkTopicEnabled) {
                return true;
            }

            return (System.currentTimeMillis() - lastMqttUpdate) < livenessTimeout;
        }

        public boolean checkReadiness() {
            if (!checkTopicEnabled) {
                return true;
            }

            return (System.currentTimeMillis() - lastMqttUpdate) < readinessTimeout;
        }

        public void close() {
            final Mqtt3BlockingClient mqtt3BlockingClient = client.toBlocking();
            if (mqtt3BlockingClient.getState().isConnected()) {
                mqtt3BlockingClient.disconnect();
            }
        }
    }
}
