/*
 * Decompiled with CFR 0.152.
 */
package reactor.core.publisher;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Iterator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Consumer;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;
import reactor.core.Exceptions;
import reactor.core.MultiProducer;
import reactor.core.Receiver;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxProcessor;
import reactor.core.publisher.Operators;
import reactor.core.publisher.RingBuffer;
import reactor.core.publisher.Wrapped;
import reactor.util.concurrent.QueueSupplier;
import reactor.util.concurrent.WaitStrategy;

abstract class EventLoopProcessor<IN>
extends FluxProcessor<IN, IN>
implements Receiver,
Runnable,
MultiProducer {
    public static final boolean TRACEABLE_RING_BUFFER_PROCESSOR = Boolean.parseBoolean(System.getProperty("reactor.ringbuffer.trace", "true"));
    static final Supplier EMITTED = Slot::new;
    final ExecutorService executor;
    final ClassLoader contextClassLoader;
    final String name;
    final boolean autoCancel;
    final RingBuffer<Slot<IN>> ringBuffer;
    final WaitStrategy readWait = WaitStrategy.liteBlocking();
    Subscription upstreamSubscription;
    volatile boolean cancelled;
    volatile int terminated;
    volatile Throwable error;
    volatile int subscriberCount;
    static final int SHUTDOWN = 1;
    static final int FORCED_SHUTDOWN = 2;
    static final AtomicIntegerFieldUpdater<EventLoopProcessor> SUBSCRIBER_COUNT = AtomicIntegerFieldUpdater.newUpdater(EventLoopProcessor.class, "subscriberCount");
    static final AtomicIntegerFieldUpdater<EventLoopProcessor> TERMINATED = AtomicIntegerFieldUpdater.newUpdater(EventLoopProcessor.class, "terminated");

    static <E> Flux<E> coldSource(RingBuffer<Slot<E>> ringBuffer, Throwable t, Throwable error, RingBuffer.Sequence start) {
        Flux bufferIterable = EventLoopProcessor.generate(start::getAsLong, (seq, sink) -> {
            long s = seq + 1L;
            if (s > ringBuffer.getCursor()) {
                sink.complete();
            } else {
                Object d = ((Slot)ringBuffer.get((long)s)).value;
                if (d != null) {
                    sink.next(d);
                }
            }
            return s;
        });
        if (error != null) {
            if (t != null) {
                t.addSuppressed(error);
                return EventLoopProcessor.concat(bufferIterable, Flux.error(t));
            }
            return EventLoopProcessor.concat(bufferIterable, Flux.error(error));
        }
        return bufferIterable;
    }

    static Runnable createRequestTask(Subscription upstream, Runnable stopCondition, Consumer<Long> postWaitCallback, LongSupplier readCount, WaitStrategy waitStrategy, Subscriber<?> errorSubscriber, int prefetch) {
        return new RequestTask(upstream, stopCondition, postWaitCallback, readCount, waitStrategy, errorSubscriber, prefetch);
    }

    static <E> RingBuffer<Slot<E>> createSingleProducer(int bufferSize) {
        return RingBuffer.createSingleProducer(EMITTED, bufferSize, WaitStrategy.busySpin());
    }

    static boolean waitRequestOrTerminalEvent(LongSupplier pendingRequest, RingBuffer.Reader barrier, AtomicBoolean isRunning, LongSupplier nextSequence, Runnable waiter) {
        try {
            while (pendingRequest.getAsLong() <= 0L) {
                long waitedSequence = nextSequence.getAsLong() + 1L;
                if (waiter != null) {
                    waiter.run();
                    barrier.waitFor(waitedSequence, waiter);
                } else {
                    barrier.waitFor(waitedSequence);
                }
                if (!isRunning.get()) {
                    throw Exceptions.failWithCancel();
                }
                LockSupport.parkNanos(1L);
            }
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            if (WaitStrategy.isAlert(e) || Exceptions.isCancel(e)) {
                return false;
            }
            throw e;
        }
        return true;
    }

    static void addCap(RingBuffer.Sequence sequence, long toAdd) {
        long u;
        long r;
        do {
            if ((r = sequence.getAsLong()) != Long.MAX_VALUE) continue;
            return;
        } while (!sequence.compareAndSet(r, u = Operators.addCap(r, toAdd)));
    }

    static long getAndSub(RingBuffer.Sequence sequence, long toSub) {
        long u;
        long r;
        do {
            if ((r = sequence.getAsLong()) != 0L && r != Long.MAX_VALUE) continue;
            return r;
        } while (!sequence.compareAndSet(r, u = Operators.subOrZero(r, toSub)));
        return r;
    }

    static RingBuffer.Sequence wrap(long init, Object delegate) {
        if (TRACEABLE_RING_BUFFER_PROCESSOR) {
            return EventLoopProcessor.wrap(RingBuffer.newSequence(init), delegate);
        }
        return RingBuffer.newSequence(init);
    }

    static RingBuffer.Sequence wrap(RingBuffer.Sequence init, Object delegate) {
        return new Wrapped<Object>(delegate, init);
    }

    EventLoopProcessor(int bufferSize, ThreadFactory threadFactory, ExecutorService executor, boolean autoCancel, boolean multiproducers, Supplier<Slot<IN>> factory, WaitStrategy strategy) {
        if (!QueueSupplier.isPowerOfTwo(bufferSize)) {
            throw new IllegalArgumentException("bufferSize must be a power of 2 : " + bufferSize);
        }
        this.autoCancel = autoCancel;
        this.contextClassLoader = new EventLoopContext();
        String name = threadFactory instanceof Supplier ? ((Supplier)((Object)threadFactory)).get().toString() : null;
        this.name = null != name ? name : this.getClass().getSimpleName();
        this.executor = executor == null ? Executors.newCachedThreadPool(threadFactory) : executor;
        this.ringBuffer = multiproducers ? RingBuffer.createMultiProducer(factory, bufferSize, strategy, this) : RingBuffer.createSingleProducer(factory, bufferSize, strategy, this);
    }

    public final boolean alive() {
        return 0 == this.terminated;
    }

    public final boolean awaitAndShutdown() {
        return this.awaitAndShutdown(-1L, TimeUnit.SECONDS);
    }

    public final boolean awaitAndShutdown(long timeout, TimeUnit timeUnit) {
        try {
            this.shutdown();
            return this.executor.awaitTermination(timeout, timeUnit);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public Flux<IN> drain() {
        return Flux.empty();
    }

    public final Flux<IN> forceShutdown() {
        int t = this.terminated;
        if (t != 2 && TERMINATED.compareAndSet(this, t, 2)) {
            this.executor.shutdownNow();
        }
        return this.drain();
    }

    public long getAvailableCapacity() {
        return this.ringBuffer.remainingCapacity();
    }

    @Override
    public Iterator<?> downstreams() {
        return Arrays.asList(this.ringBuffer.getSequenceReceivers()).iterator();
    }

    @Override
    public final Throwable getError() {
        return this.error;
    }

    @Override
    public final String toString() {
        return "/Processors/" + this.name + "/" + this.contextClassLoader.hashCode();
    }

    public final int hashCode() {
        return this.contextClassLoader.hashCode();
    }

    @Override
    public final boolean isCancelled() {
        return this.cancelled;
    }

    @Override
    public final boolean isStarted() {
        return this.upstreamSubscription != null || this.ringBuffer.getAsLong() != -1L;
    }

    @Override
    public final boolean isTerminated() {
        return this.terminated > 0;
    }

    public final void onComplete() {
        if (TERMINATED.compareAndSet(this, 0, 1)) {
            this.upstreamSubscription = null;
            this.executor.shutdown();
            this.readWait.signalAllWhenBlocking();
            this.doComplete();
        }
    }

    public final void onError(Throwable t) {
        if (t == null) {
            throw Exceptions.argumentIsNullException();
        }
        if (TERMINATED.compareAndSet(this, 0, 1)) {
            this.error = t;
            this.upstreamSubscription = null;
            this.executor.shutdown();
            this.readWait.signalAllWhenBlocking();
            this.doError(t);
        }
    }

    public final void onNext(IN o) {
        if (o == null) {
            throw Exceptions.argumentIsNullException();
        }
        long seqId = this.ringBuffer.next();
        Slot<IN> signal = this.ringBuffer.get(seqId);
        signal.value = o;
        this.ringBuffer.publish(seqId);
    }

    public final void onSubscribe(Subscription s) {
        if (Operators.validate(this.upstreamSubscription, s)) {
            this.upstreamSubscription = s;
            try {
                if (s != Operators.emptySubscription()) {
                    this.requestTask(s);
                }
            }
            catch (Throwable t) {
                this.onError(Operators.onOperatorError(s, t));
            }
        }
    }

    public final void shutdown() {
        try {
            this.onComplete();
            this.executor.shutdown();
        }
        catch (Throwable t) {
            this.onError(Operators.onOperatorError(t));
        }
    }

    public final Subscription upstream() {
        return this.upstreamSubscription;
    }

    @Override
    public final long getCapacity() {
        return this.ringBuffer.bufferSize();
    }

    final void cancel() {
        this.cancelled = true;
        if (TERMINATED.compareAndSet(this, 0, 1)) {
            this.executor.shutdown();
        }
        this.readWait.signalAllWhenBlocking();
    }

    protected void doComplete() {
    }

    protected void requestTask(Subscription s) {
    }

    final void decrementSubscribers() {
        Subscription subscription = this.upstreamSubscription;
        int subs = SUBSCRIBER_COUNT.decrementAndGet(this);
        if (subs == 0 && subscription != null && this.autoCancel) {
            this.upstreamSubscription = null;
            this.cancel();
        }
    }

    abstract void doError(Throwable var1);

    final boolean incrementSubscribers() {
        return SUBSCRIBER_COUNT.getAndIncrement(this) == 0;
    }

    final boolean startSubscriber(Subscriber<? super IN> subscriber, Subscription subscription) {
        try {
            Thread.currentThread().setContextClassLoader(this.contextClassLoader);
            subscriber.onSubscribe(subscription);
            return true;
        }
        catch (Throwable t) {
            Operators.error(subscriber, t);
            return false;
        }
    }

    static final class EventLoopFactory
    implements ThreadFactory,
    Supplier<String> {
        static final AtomicInteger COUNT = new AtomicInteger();
        final String name;
        final boolean daemon;

        EventLoopFactory(String name, boolean daemon) {
            this.name = name;
            this.daemon = daemon;
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r, this.name + "-" + COUNT.incrementAndGet());
            t.setDaemon(this.daemon);
            return t;
        }

        @Override
        public String get() {
            return this.name;
        }
    }

    public static final class Slot<T>
    implements Serializable {
        private static final long serialVersionUID = 5172014386416785095L;
        public T value;
    }

    static final class EventLoopContext
    extends ClassLoader {
        EventLoopContext() {
            super(Thread.currentThread().getContextClassLoader());
        }
    }

    static final class RequestTask
    implements Runnable {
        final WaitStrategy waitStrategy;
        final LongSupplier readCount;
        final Subscription upstream;
        final Runnable spinObserver;
        final Consumer<Long> postWaitCallback;
        final Subscriber<?> errorSubscriber;
        final int prefetch;

        public RequestTask(Subscription upstream, Runnable stopCondition, Consumer<Long> postWaitCallback, LongSupplier readCount, WaitStrategy waitStrategy, Subscriber<?> errorSubscriber, int prefetch) {
            this.waitStrategy = waitStrategy;
            this.readCount = readCount;
            this.postWaitCallback = postWaitCallback;
            this.errorSubscriber = errorSubscriber;
            this.upstream = upstream;
            this.spinObserver = stopCondition;
            this.prefetch = prefetch;
        }

        @Override
        public void run() {
            long bufferSize = this.prefetch;
            long limit = bufferSize - Math.max(bufferSize >> 2, 1L);
            long cursor = -1L;
            try {
                this.spinObserver.run();
                this.upstream.request(bufferSize);
                while (true) {
                    long c = cursor + limit;
                    cursor = this.waitStrategy.waitFor(c, this.readCount, this.spinObserver);
                    if (this.postWaitCallback != null) {
                        this.postWaitCallback.accept(cursor);
                    }
                    this.upstream.request(limit + (cursor - c));
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Throwable t) {
                if (Exceptions.isCancel(t)) {
                    this.upstream.cancel();
                    return;
                }
                if (WaitStrategy.isAlert(t)) {
                    return;
                }
                this.errorSubscriber.onError(Operators.onOperatorError(t));
            }
        }
    }
}

