package io.privy.auth.oAuth

import io.privy.auth.PrivyUser

/**
 * Interface for authenticating users via OAuth providers.
 * This interface provides a simplified API for applications to implement OAuth-based
 * authentication flows with various providers like Google, Apple, etc.
 */
public interface LoginWithOAuth {
    /**
     * Initiates the OAuth login flow with the specified provider by launching the provider's
     * authentication page in a chrome tab. After the authentication is complete, the chrome tab
     * will redirect back to your app using the "appUrlScheme" provided. This will trigger your
     * Activity's "onNewIntent" or "onCreate", depending on the "android:launchMode" set in your Manifest.
     * Once you receive the intent from the redirect, grab the "intent.dataString", and pass it to Privy
     * via [handleIntentData]. If Privy can handle the redirect, we will authenticate the user, and
     * invoke the [@param onComplete].
     *
     * @param oAuthProvider The OAuth provider to authenticate with (e.g., Google, Twitter, Discord)
     * @param appUrlScheme The URL scheme registered for the app to handle redirects
     * @param onComplete Callback that will be invoked with the result of the login attempt
     */
    public suspend fun login(
        oAuthProvider: OAuthProvider,
        appUrlScheme: String,
        onComplete: (Result<PrivyUser>) -> Unit
    )

    /**
     * Handles the OAuth redirect URL to complete the login flow. You may call [canHandleIntentData]
     * to determine if Privy can handle the redirect.
     *
     * @param intentDataString The redirect URL received by the application after the Oauth flow redirects
     * back to the app, which is accessible via "intent.dataString". See [login] for more details.
     *
     * @return Boolean indicating whether the redirect was successfully handled.
     *   If true, the redirect was processed and the authentication flow is continuing.
     *   If false, the redirect could not be handled (invalid URL, no pending request, etc.).
     */
    public fun handleIntentData(intentDataString: String): Boolean

    /**
     * A helper method that allows you to check if Privy can handle the redirect URL.
     *
     * @param intentDataString The redirect URL received by the application after the Oauth flow redirects
     * back to the app, which is accessible via "intent.dataString". See [login] for more details.
     *
     * @return A boolean representing if the URL can be handled. If true, it is safe to call [handleIntentData]
     */
    public fun canHandleIntentData(intentDataString: String): Boolean
}