package io.privy.auth.sms

import io.privy.auth.AuthRepository
import io.privy.auth.internal.InternalAuthManager
import io.privy.auth.AuthenticationException
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.otp.OtpRequestType
import io.privy.logging.PrivyLogger
import io.privy.network.toResult
import me.tatarka.inject.annotations.Inject

@Inject
public class RealLoginWithSms(
  private val authRepository: AuthRepository,
  private val internalAuthManager: InternalAuthManager,
  private val privyLogger: PrivyLogger,
) : LoginWithSms {
  // cache phone number otp was sent to
  private var phoneNumberOtpSentTo: String? = null

  override suspend fun sendCode(phoneNumber: String): Result<Unit> {
    return authRepository
      .sendOtp(otpRequestType = OtpRequestType.Phone(phoneNumber))
      .toResult()
      .onSuccess {
        // Cache phone number OTP was sent to
        this.phoneNumberOtpSentTo = phoneNumber
      }
      .onFailure {
        privyLogger.error("Error sending OTP: ${it.message}")
      }
  }

  override suspend fun loginWithCode(code: String, phoneNumber: String?): Result<PrivyUser> {
    // if phone number specified in method parameters, use it, else fall back to cached
    val sentToPhoneNumber =
        phoneNumber
            ?: phoneNumberOtpSentTo
                ?: return Result.failure(AuthenticationException("Phone number not specified."))

    return internalAuthManager.login(
        loginType = LoginType.Sms(phoneNumber = sentToPhoneNumber, code = code),
      )
  }
}
