package io.privy.auth.session

import kotlinx.serialization.json.Json
import kotlinx.serialization.modules.SerializersModule
import kotlinx.serialization.modules.polymorphic
import me.tatarka.inject.annotations.Inject

public interface AuthSessionResponseDeserializer {
    public fun deserialize(jsonResponse: String): RawAuthSessionResponse

    public fun deserializeUser(jsonResponse: String): RefreshUserResponse

    public fun deserializeRawUser(jsonResponse: String): PrivyUserResponse
}

@Inject
public class RealAuthSessionResponseDeserializer : AuthSessionResponseDeserializer {
    private val authSessionSerializer = Json {
        prettyPrint = true
        isLenient = true
        ignoreUnknownKeys = true
        explicitNulls = false

        serializersModule = SerializersModule {
            polymorphic(LinkedAccountResponse::class) {
                subclass(
                    LinkedAccountResponse.WalletAccount::class,
                    LinkedAccountResponse.WalletAccount.serializer()
                )
                subclass(
                    LinkedAccountResponse.CustomAuth::class,
                    LinkedAccountResponse.CustomAuth.serializer()
                )
                subclass(
                    LinkedAccountResponse.PhoneNumberAccount::class,
                    LinkedAccountResponse.PhoneNumberAccount.serializer()
                )
                subclass(
                    LinkedAccountResponse.EmailAddressAccount::class,
                    LinkedAccountResponse.EmailAddressAccount.serializer()
                )
                subclass(
                    LinkedAccountResponse.GoogleOAuthAccount::class,
                    LinkedAccountResponse.GoogleOAuthAccount.serializer()
                )
                subclass(
                    LinkedAccountResponse.TwitterOAuthAccount::class,
                    LinkedAccountResponse.TwitterOAuthAccount.serializer()
                )
                subclass(
                    LinkedAccountResponse.DiscordOAuthAccount::class,
                    LinkedAccountResponse.DiscordOAuthAccount.serializer()
                )

                defaultDeserializer { LinkedAccountResponse.UnknownAccount.serializer() }
            }

            classDiscriminator = "type"
        }
    }

    override fun deserialize(jsonResponse: String): RawAuthSessionResponse {
        return authSessionSerializer.decodeFromString<RawAuthSessionResponse>(jsonResponse)
    }

    override fun deserializeUser(jsonResponse: String): RefreshUserResponse {
        return authSessionSerializer.decodeFromString<RefreshUserResponse>(jsonResponse)
    }

    override fun deserializeRawUser(jsonResponse: String): PrivyUserResponse {
        return authSessionSerializer.decodeFromString<PrivyUserResponse>(jsonResponse)
    }
}
