package io.privy.auth.oAuth

import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

/**
 * Singleton class that manages OAuth data sharing between LoginWithOAuth and PrivyRedirectActivity.
 * This centralizes the state management for OAuth flows, eliminating the need for static properties
 * scattered across different classes.
 */
public object OAuthDataManager {

  // Continuation for awaitCustomTabResult
  private var customTabContinuation: Continuation<Result<String>>? = null

  /**
   * Suspends the coroutine until the custom tab result is available. This function stores the
   * continuation and resumes it when the activity calls handleCustomTabResult.
   */
  public suspend fun awaitCustomTabResult(): Result<String> {
    // If there's already an ongoing OAuth flow, cancel it first
    customTabContinuation?.let { existing ->
      existing.resume(Result.failure(Exception("OAuth flow cancelled - new login initiated")))
      customTabContinuation = null
    }
    
    return suspendCoroutine { continuation -> 
      customTabContinuation = continuation
    }
  }

  /**
   * Handles the custom tab result from the activity. This resumes the continuation stored by
   * awaitCustomTabResult.
   */
  public fun handleCustomTabResult(result: Result<String>) {
    customTabContinuation?.resume(result)
    customTabContinuation = null
  }
}
