package io.privy.auth.session.internal

import io.privy.auth.internal.InternalPrivyUser
import io.privy.auth.internal.LoginMethod
import io.privy.auth.session.RawAuthSessionResponse
import io.privy.logging.PrivyLogger
import me.tatarka.inject.annotations.Inject

public interface MapRawAuthSessionResponseToAuthSessionResponse {
  public operator fun invoke(
    rawAuthSessionResponse: RawAuthSessionResponse,
    loginMethod: LoginMethod,
  ): AuthSessionResponse
}

@Inject
public class RealMapRawAuthSessionResponseToAuthSessionResponse(
  private val logger: PrivyLogger,
) : MapRawAuthSessionResponseToAuthSessionResponse {
  override operator fun invoke(
    rawAuthSessionResponse: RawAuthSessionResponse,
    loginMethod: LoginMethod
  ): AuthSessionResponse {
    return with(rawAuthSessionResponse) {
      val sessionTokens = toAuthSessionResponseTokensOrNull()

      val tokenInfo: SessionUpdateAction = when(sessionUpdateAction.lowercase()) {
        "set" -> {
          // Backend is telling us the auth response is a valid, and we should store session tokens
          // This means backend should have properly sent session tokens
          if (sessionTokens == null) {
            // Backend should always specify tokens when sessionUpdateAction == set -- so this should NEVER happen.
            logger.error("Privy backend failed to return session tokens. Please reach out to customer support.")

            // Catastrophic error - so just log user out
            SessionUpdateAction.Clear
          } else {
            SessionUpdateAction.Set(sessionTokens)
          }
        }
        "clear" -> {
          // Privy backend wants client to log user out
          SessionUpdateAction.Clear
        }
        "ignore" -> {
          // Privy backend thinks this might be a duplicate request, but there are some cases we do
          // want to store the tokens
          SessionUpdateAction.Ignore(sessionTokens)
        }
        else -> {
          // This should never happen
          logger.error("Unexpected session update action received. Please reach out to customer support.")

          // Catastrophic error - so just log user out
          SessionUpdateAction.Clear
        }
      }

     AuthSessionResponse(
       user = user.toInternalPrivyUser(),
       sessionUpdateAction = tokenInfo,
       loginMethod = loginMethod,
     )
    }
  }

  private fun RawAuthSessionResponse.toAuthSessionResponseTokensOrNull(): AuthSessionResponseTokens? {
    return if (token.isNullOrEmpty() || refreshToken.isNullOrEmpty()) {
      // Access token and refresh token must be specified to be considered valid
      null
    } else {
      AuthSessionResponseTokens(
        token = token,
        refreshToken = refreshToken,
        identityToken = identityToken,
      )
    }
  }
}

public data class AuthSessionResponse(
  val user: InternalPrivyUser,
  val sessionUpdateAction: SessionUpdateAction,
  val loginMethod: LoginMethod,
)

public sealed interface SessionUpdateAction {
  public data object Clear: SessionUpdateAction

  public data class Set(val authSessionResponseTokens: AuthSessionResponseTokens): SessionUpdateAction

  // There are certain cases where "ignore" sessionUpdateAction will return valid tokens
  // If there are valid tokens, we should store the session
  public data class Ignore(val authSessionResponseTokens: AuthSessionResponseTokens?): SessionUpdateAction
}

public data class AuthSessionResponseTokens(
  val token: String,
  val refreshToken: String,
  val identityToken: String?,
)
