package io.privy.auth.email

import io.privy.auth.AuthRepository
import io.privy.auth.internal.InternalAuthManager
import io.privy.auth.AuthenticationException
import io.privy.auth.LoginType
import io.privy.auth.PrivyUser
import io.privy.auth.otp.OtpRequestType
import io.privy.logging.PrivyLogger
import io.privy.network.toResult
import me.tatarka.inject.annotations.Inject

@Inject
public class RealLoginWithEmail(
  private val authRepository: AuthRepository,
  private val internalAuthManager: InternalAuthManager,
  private val privyLogger: PrivyLogger,
) : LoginWithEmail {
  private var emailOtpSentTo: String? = null

  override suspend fun sendCode(email: String): Result<Unit> {
    return authRepository
      .sendOtp(otpRequestType = OtpRequestType.Email(email))
      .toResult()
      .onSuccess {
        // Cache email OTP was sent to
        this.emailOtpSentTo = email
      }
      .onFailure {
        privyLogger.error("Error sending OTP: ${it.message}")
      }
  }

  override suspend fun loginWithCode(code: String, email: String?): Result<PrivyUser> {
    // if email is specified in method parameters, use it, else fall back to cached
    val sentToEmail =
        email
            ?: emailOtpSentTo
                ?: return Result.failure(AuthenticationException("Email address not specified."))

    return internalAuthManager.login(loginType = LoginType.Email(emailAddress = sentToEmail, code = code))
  }

  override suspend fun linkWithCode(code: String, email: String?): Result<Unit> {
    val sentToEmail =
        email
            ?: emailOtpSentTo
                ?: return Result.failure(AuthenticationException("Email address not specified."))

    return internalAuthManager.linkAccount(
        loginType = LoginType.Email(emailAddress = sentToEmail, code = code))
  }
}
