package io.privy.appconfig

import io.ktor.resources.Resource
import io.ktor.util.logging.Logger
import io.privy.di.PrivyAppId
import io.privy.logging.PrivyLogger
import io.privy.network.AuthType
import io.privy.network.KtorWrapper
import io.privy.network.toResult
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import me.tatarka.inject.annotations.Inject

@Inject
public class RealAppConfigRepository(
  private val ktorWrapper: KtorWrapper,
  private val appId: PrivyAppId,
  private val privyLogger: PrivyLogger,
) : AppConfigRepository {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)
  private val mutex = Mutex()
  private var cachedAppConfig: AppConfig? = null

  init {
    // Proactively fetch app config so it's hopefully cached by the first call
    scope.launch {
      loadAppConfig()
    }
  }

  override suspend fun loadAppConfig(): Result<AppConfig> {
    mutex.withLock {
      // if config is previously cached, return it
      cachedAppConfig?.let { return Result.success(it) }

      return ktorWrapper
        .getResult<AppConfigResource, AppConfig>(
          resource = AppConfigResource(appId = appId),
          authType = AuthType.None,
        )
        .toResult()
        // cache result on successful fetch
        .onSuccess {
          privyLogger.internal("Fetched app configuration: $it}")
          this.cachedAppConfig = it
        }
    }
  }

  @Resource("apps/{appId}")
  private class AppConfigResource(val appId: String)
}
