package io.primer.android

import android.net.Uri
import io.primer.android.logging.DefaultLogger
import io.primer.android.payment.WebBrowserIntentBehaviour
import java.util.UUID

internal class WebviewInteropRegister {

    data class Callback(
        val id: String,
        val cancelUrl: String,
        val successUrl: String,
        val behaviour: WebBrowserIntentBehaviour,
        var result: Uri? = null,
    )

    companion object {

        private val log = DefaultLogger("WebviewInteropActivity")
        private val callbacks: MutableMap<String, Callback> = mutableMapOf()
        private lateinit var scheme: String

        fun init(scheme: String) {
            log.info("Initializing: $scheme")
            this.scheme = scheme
        }

        fun register(behaviour: WebBrowserIntentBehaviour): Callback {
            val id = UUID.randomUUID().toString()
            val callback = Callback(
                id = id,
                cancelUrl = "$scheme://$id/cancel",
                successUrl = "$scheme://$id/success",
                behaviour = behaviour
            )

            callbacks[id] = callback

            return callback
        }

        fun handleResult(uri: Uri?) {
            val callback = callbacks[uri?.host]

            if (callback != null) {
                callback.result = uri
            }
        }

        fun invokeAll() {
            callbacks.forEach {
                it.value.result.let { uri ->
                    when (uri?.pathSegments?.last()) {
                        "success" -> it.value.behaviour.onSuccess(uri)
                        "cancel" -> it.value.behaviour.onCancel(uri)
                        null -> it.value.behaviour.onCancel()
                    }
                }

                if (it.value.result == null) {
                    it.value.behaviour.onCancel()
                }
            }

            callbacks.clear()
        }
    }
}
