package io.primer.android.payment.card

import android.content.res.ColorStateList
import android.graphics.drawable.RippleDrawable
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.DrawableCompat
import io.primer.android.ui.settings.PrimerTheme
import io.primer.android.R
import io.primer.android.PrimerSessionIntent
import io.primer.android.data.configuration.models.PaymentMethodRemoteConfig
import io.primer.android.databinding.PaymentMethodButtonCardBinding
import io.primer.android.di.DIAppComponent
import io.primer.android.data.settings.internal.PrimerConfig
import io.primer.android.model.SyncValidationError
import io.primer.android.payment.NewFragmentBehaviour
import io.primer.android.payment.PaymentMethodDescriptor
import io.primer.android.payment.PaymentMethodUiType
import io.primer.android.payment.SelectedPaymentMethodBehaviour
import io.primer.android.payment.VaultCapability
import io.primer.android.payment.utils.ButtonViewHelper.generateButtonContent
import io.primer.android.ui.CardNumberFormatter
import io.primer.android.ui.ExpiryDateFormatter
import io.primer.android.ui.fragments.CardFormFragment
import org.json.JSONObject
import org.koin.core.component.KoinApiExtension
import org.koin.core.component.inject

internal const val CARD_NAME_FILED_NAME = "cardholderName"
internal const val CARD_NUMBER_FIELD_NAME = "number"
internal const val CARD_EXPIRY_FIELD_NAME = "date"
internal const val CARD_CVV_FIELD_NAME = "cvv"
internal const val CARD_POSTAL_CODE_FIELD_NAME = "postalCode"
internal const val CARD_EXPIRY_MONTH_FIELD_NAME = "expirationMonth"
internal const val CARD_EXPIRY_YEAR_FIELD_NAME = "expirationYear"

@KoinApiExtension
internal class CreditCard(
    config: PaymentMethodRemoteConfig,
    private val options: Card,
    encodedAsJson: JSONObject = JSONObject(), // FIXME passing in a as dependency so we can test
) : PaymentMethodDescriptor(config, encodedAsJson), DIAppComponent {

    private val checkoutConfig: PrimerConfig by inject()
    private val theme: PrimerTheme by inject()

    var hasPostalCode: Boolean = false
    var hasCardholderName: Boolean = true

    // FIXME static call + instantiation makes it impossible to properly test
    override val selectedBehaviour: SelectedPaymentMethodBehaviour
        get() = NewFragmentBehaviour(CardFormFragment::newInstance, returnToPreviousOnBack = true)

    override val behaviours: List<SelectedPaymentMethodBehaviour> = emptyList()

    override val type: PaymentMethodUiType = PaymentMethodUiType.FORM

    override val vaultCapability: VaultCapability = VaultCapability.SINGLE_USE_AND_VAULT

    override fun createButton(container: ViewGroup): View {
        val binding = PaymentMethodButtonCardBinding.inflate(
            LayoutInflater.from(container.context),
            container,
            false
        )

        val content = generateButtonContent(theme, container.context)
        val splash = theme.splashColor.getColor(container.context, theme.isDarkMode)
        val rippleColor = ColorStateList.valueOf(splash)
        binding.cardPreviewButton.background = RippleDrawable(rippleColor, content, null)

        val text = binding.cardPreviewButtonText
        val drawable = ContextCompat.getDrawable(
            container.context,
            R.drawable.ic_logo_credit_card
        )

        text.setCompoundDrawablesWithIntrinsicBounds(drawable, null, null, null)

        text.setTextColor(
            theme.paymentMethodButton.text.defaultColor.getColor(
                container.context,
                theme.isDarkMode
            )
        )

        text.text = when (checkoutConfig.paymentMethodIntent) {
            PrimerSessionIntent.CHECKOUT -> container.context.getString(R.string.pay_by_card)
            PrimerSessionIntent.VAULT ->
                container.context.getString(R.string.credit_debit_card)
        }

        val icon = text.compoundDrawables

        DrawableCompat.setTint(
            DrawableCompat.wrap(icon[0]),
            theme.paymentMethodButton.text.defaultColor.getColor(
                container.context,
                theme.isDarkMode
            )
        )

        return binding.root
    }

    // FIXME a model should not be responsible for parsing itself into json
    override fun toPaymentInstrument(): JSONObject {
        val json = JSONObject()

        json.put(CARD_NAME_FILED_NAME, getStringValue(CARD_NAME_FILED_NAME).trim())
        json.put(
            CARD_NUMBER_FIELD_NAME,
            getStringValue(CARD_NUMBER_FIELD_NAME).replace("\\s".toRegex(), "")
        )
        json.put(CARD_CVV_FIELD_NAME, getStringValue(CARD_CVV_FIELD_NAME))

        val expiry = ExpiryDateFormatter.fromString(getStringValue(CARD_EXPIRY_FIELD_NAME))

        json.put(CARD_EXPIRY_MONTH_FIELD_NAME, expiry.getMonth())
        json.put(CARD_EXPIRY_YEAR_FIELD_NAME, expiry.getYear())

        return json
    }

    // FIXME this should not be here. a model should not be responsible for validating itself
    override fun validate(): List<SyncValidationError> {
        val errors = ArrayList<SyncValidationError>()

        if (hasCardholderName) {
            val name = getSanitizedValue(CARD_NAME_FILED_NAME)
            if (name.isEmpty()) {
                errors.add(
                    SyncValidationError(
                        name = CARD_NAME_FILED_NAME,
                        errorId = R.string.form_error_required,
                        fieldId = R.string.card_holder_name
                    )
                )
            }
        }

        // FIXME static call (formatter should be injected)
        val number = CardNumberFormatter.fromString(getSanitizedValue(CARD_NUMBER_FIELD_NAME))

        if (number.isEmpty()) {
            errors.add(
                SyncValidationError(
                    name = CARD_NUMBER_FIELD_NAME,
                    errorId = R.string.form_error_required,
                    fieldId = R.string.card_number
                )
            )
        } else if (!number.isValid()) {
            errors.add(
                SyncValidationError(
                    name = CARD_NUMBER_FIELD_NAME,
                    errorId = R.string.form_error_invalid,
                    fieldId = R.string.card_number
                )
            )
        }

        val cvv = getSanitizedValue(CARD_CVV_FIELD_NAME)

        if (cvv.isEmpty()) {
            errors.add(
                SyncValidationError(
                    name = CARD_CVV_FIELD_NAME,
                    errorId = R.string.form_error_required,
                    fieldId = R.string.card_cvv
                )
            )
        } else if (cvv.length != number.getCvvLength()) {
            errors.add(
                SyncValidationError(
                    name = CARD_CVV_FIELD_NAME,
                    errorId = R.string.form_error_invalid,
                    fieldId = R.string.card_cvv
                )
            )
        }

        // FIXME static call (formatter should be injected)
        val expiry = ExpiryDateFormatter.fromString(getSanitizedValue(CARD_EXPIRY_FIELD_NAME))

        if (expiry.isEmpty()) {
            errors.add(
                SyncValidationError(
                    name = CARD_EXPIRY_FIELD_NAME,
                    errorId = R.string.form_error_required,
                    fieldId = R.string.card_expiry
                )
            )
        } else if (!expiry.isValid()) {
            errors.add(
                SyncValidationError(
                    name = CARD_EXPIRY_FIELD_NAME,
                    errorId = R.string.form_error_invalid,
                    fieldId = R.string.card_expiry
                )
            )
        }

        if (hasPostalCode) {
            val postalCode = getSanitizedValue(CARD_POSTAL_CODE_FIELD_NAME)

            if (postalCode.isEmpty()) {
                errors.add(
                    SyncValidationError(
                        name = CARD_POSTAL_CODE_FIELD_NAME,
                        errorId = R.string.form_error_required,
                        fieldId = R.string.card_zip
                    )
                )
            }
        }

        return errors
    }

    override fun getValidAutoFocusableFields(): Set<String> {
        val fields = hashSetOf<String>()
        val number = CardNumberFormatter.fromString(getSanitizedValue(CARD_NUMBER_FIELD_NAME))
        if (number.isValid() && number.getMaxLength() == number.getValue().length) {
            fields.add(CARD_NUMBER_FIELD_NAME)
        }

        val cvv = getSanitizedValue(CARD_CVV_FIELD_NAME)
        if (cvv.length == number.getCvvLength()) fields.add(CARD_CVV_FIELD_NAME)

        val expiry = ExpiryDateFormatter.fromString(getSanitizedValue(CARD_EXPIRY_FIELD_NAME))
        if (expiry.isValid()) fields.add(CARD_EXPIRY_FIELD_NAME)

        return fields
    }

    private fun getSanitizedValue(key: String): String {
        return getStringValue(key).trim()
    }
}
