package io.poyarzun.concoursedsl.printer

import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import io.poyarzun.concoursedsl.domain.*
import io.poyarzun.concoursedsl.dsl.*
import io.poyarzun.concoursedsl.exhaustivePipeline
import io.poyarzun.concoursedsl.resources.get
import io.poyarzun.concoursedsl.resources.gitResource
import java.io.File
import kotlin.reflect.KClass
import kotlin.reflect.KMutableProperty1
import kotlin.reflect.KProperty1
import kotlin.reflect.full.declaredMemberProperties


fun main() {
    val p = exhaustivePipeline

    val file = FileSpec.builder("pipeline", "Main")
            .addImport("io.poyarzun.concoursedsl.dsl",
                    "pipeline",
                    "inputMapping"
            )
            .addFunction(pipelineToSource(p))
            .addFunction(FunSpec.builder("main")
                    .addStatement("println(mainPipeline())")
                    .build())
            .build()

    val f = File("printer-test/src/main/kotlin")
    f.deleteRecursively()
    file.writeTo(f)
}


fun pipelineToSource(pipeline: Pipeline): FunSpec {
    return FunSpec.builder("mainPipeline")
            .beginControlFlow("return pipeline")
            .resources(pipeline.resources)
            .jobs(pipeline.jobs)
            .endControlFlow()
            .build()
}

fun FunSpec.Builder.resources(resources: MutableList<Resource<Any>>): FunSpec.Builder {
    fun FunSpec.Builder.resourceToSource(resource: Resource<Any>) {
        beginControlFlow("resource(%S, %S)", resource.name, resource.type)
                .beginControlFlow("source")
                .configurePlainObject(resource.source)
                .endControlFlow()
                .configurePlainObject(resource)
                .endControlFlow()
    }

    resources.forEach(::resourceToSource)
    return this
}

fun FunSpec.Builder.jobs(jobs: MutableList<Job>): FunSpec.Builder {
    fun FunSpec.Builder.jobToSource(job: Job) {
        beginControlFlow("job(%S)", job.name)
                .beginControlFlow("plan")
                .steps(job.plan)
                .endControlFlow()
                .configurePlainObject(job)
                .endControlFlow()
    }

    jobs.forEach(::jobToSource)
    return this
}

fun FunSpec.Builder.steps(steps: MutableList<Step>): FunSpec.Builder {
    fun FunSpec.Builder.stepToSource(step: Step) {
        when (step) {
            is Step.GetStep<*> -> {
                beginControlFlow("get(%S)", step.get)
                        .beginControlFlow("params")
                        .configurePlainObject(step.params)
                        .endControlFlow()
                        .configurePlainObject(step)
                        .endControlFlow()
            }
            is Step.PutStep<*, *> -> {
                beginControlFlow("put(%S)", step.put)
                        .beginControlFlow("params")
                        .configurePlainObject(step.params)
                        .endControlFlow()
                        .beginControlFlow("getParams")
                        .configurePlainObject(step.getParams)
                        .endControlFlow()
                        .configurePlainObject(step)
                        .endControlFlow()
            }
            is Step.TaskStep -> {
                beginControlFlow("task(%S)", step.task)
                        .configurationBlock("inputMapping", step.inputMapping)
                        .configurationBlock("outputMapping", step.outputMapping)
                        .configurationBlock("config", step.config)
                        .configurePlainObject(step, "config")
                        .endControlFlow()
            }
        }
    }

    steps.forEach(::stepToSource)
    return this
}

inline fun <reified T : Any> FunSpec.Builder.configurationBlock(name: String, value: T?): FunSpec.Builder {
    if (value == null) {
        return this
    }

    return beginControlFlow(name)
            .configurePlainObject(value)
            .endControlFlow()
}


inline fun <reified T : Any> FunSpec.Builder.configurePlainObject(value: T, vararg skipList: String) =
        configurePlainObject(value, T::class, skipList)


// configurePlainObject emits an assignment for each mutable property in value
fun <T : Any> FunSpec.Builder.configurePlainObject(value: T, type: KClass<T>, skipList: Array<out String>): FunSpec.Builder {
    when (value) {
        is Map<*, *> -> {
            value.forEach {
                val key = it.key
                val value = it.value
                when (key) {
                    is String ->
                        when (value) {
                            is String ->
                                addStatement("put(%S, %S)", key, value)
                            else ->
                                addStatement("put(%S, TODO(%S))", key, "Could not automatically set value: $value")
                        }
                    else ->
                        when (value) {
                            is String ->
                                addStatement("put(TODO(%S), %S)", "Could not automatically set value: $key", value)
                            else ->
                                addStatement("put(TODO(%S), TODO(%S))", "Could not automatically set value: $key", "Could not automatically set value: $value")
                        }
                }
            }
        }
        else -> {
            if (type == Any::class) {
                addStatement("TODO(%S)", "Could not automatically configurePlainObject ${value.javaClass}")
                return this
            }
            type.declaredMemberProperties.forEach { prop: KProperty1<T, *> ->
                if (prop !is KMutableProperty1<T, *> || skipList.contains(prop.name)) return@forEach

                val propValue = prop.get(value)
                if (propValue != null) {
                    when (propValue) {
                        is String ->
                            addStatement("${prop.name} = %S", propValue)
                        is Int ->
                            addStatement("${prop.name} = %L", propValue)
                        is Boolean ->
                            addStatement("${prop.name} = %L", propValue)
                        else -> {
                            addStatement("${prop.name} = TODO(%S)", "Could not automatically generate type ${prop.returnType}")
                        }
                    }
                }
            }
        }
    }

    return this
}

