package io.polywrap.wasm

import io.polywrap.core.Invoker
import io.polywrap.core.Wrapper
import uniffi.polywrap_native.FfiInvoker
import uniffi.polywrap_native.FfiWrapper
import uniffi.polywrap_native.ffiWrapperFromBytecode

/**
 * Represents a WebAssembly (Wasm) wrapper for executing Wasm code.
 *
 * @param ffiWrapper The underlying FFI wrapper instance.
 */
@OptIn(ExperimentalUnsignedTypes::class)
data class WasmWrapper(val ffiWrapper: FfiWrapper) : Wrapper, AutoCloseable {

    /**
     * Creates a new [WasmWrapper] instance with the given Wasm module.
     *
     * @param wasmModule The WebAssembly module as a ByteArray.
     */
    constructor(wasmModule: ByteArray) : this(
        ffiWrapperFromBytecode(wasmModule.asUByteArray().asList())
    )

    override fun invoke(
        method: String,
        args: List<UByte>?,
        env: List<UByte>?,
        invoker: FfiInvoker
    ): List<UByte> = ffiWrapper.invoke(method, args, env, invoker)

    override fun invoke(
        method: String,
        args: ByteArray?,
        env: ByteArray?,
        invoker: Invoker
    ): Result<ByteArray> = runCatching {
        ffiWrapper.invoke(
            method = method,
            args = args?.asUByteArray()?.toList(),
            env = env?.asUByteArray()?.toList(),
            invoker = invoker.ffiInvoker
        ).toUByteArray().asByteArray()
    }

    override fun close() = ffiWrapper.close()
}
