package io.polywrap.configBuilder

import io.polywrap.core.WrapPackage
import io.polywrap.core.Wrapper
import io.polywrap.wasm.WasmPackage
import io.polywrap.wasm.WasmWrapper
import uniffi.polywrap_native.FfiBuilderConfig
import uniffi.polywrap_native.FfiClient
import uniffi.polywrap_native.FfiWrapPackage
import uniffi.polywrap_native.FfiWrapper
import uniffi.polywrap_native.ffiUriFromString

@OptIn(ExperimentalUnsignedTypes::class)
internal class FfiConfigBuilder : AutoCloseable {

    private val ffiBuilderConfig: FfiBuilderConfig = FfiBuilderConfig()

    fun addEnv(uri: String, env: ByteArray) {
        ffiUriFromString(uri).use { ffiUri ->
            ffiBuilderConfig.addEnv(ffiUri, env.asUByteArray().toList())
        }
    }

    fun removeEnv(uri: String) {
        ffiUriFromString(uri).use { ffiUri ->
            ffiBuilderConfig.removeEnv(ffiUri)
        }
    }

    fun addInterfaceImplementation(interfaceUri: String, implementationUri: String) {
        ffiUriFromString(interfaceUri).use { ffiInterfaceUri ->
            ffiUriFromString(implementationUri).use { ffiImplementationUri ->
                ffiBuilderConfig.addInterfaceImplementation(ffiInterfaceUri, ffiImplementationUri)
            }
        }
    }

    fun removeInterfaceImplementation(interfaceUri: String, implementationUri: String) {
        ffiUriFromString(interfaceUri).use { ffiInterfaceUri ->
            ffiUriFromString(implementationUri).use { ffiImplementationUri ->
                ffiBuilderConfig.removeInterfaceImplementation(ffiInterfaceUri, ffiImplementationUri)
            }
        }
    }

    /**
     * Adds a wrapper with a specified URI key to the current configuration.
     *
     * @param wrapper A [Pair] of the URI key and the [Wrapper] to add.
     * @return This [BaseConfigBuilder] instance for chaining calls.
     */
    fun addWrapper(uri: String, wrapper: Wrapper) {
        ffiUriFromString(uri).use { ffiUri -> 
            val ffiWrapper = when (wrapper is WasmWrapper) {
                true -> wrapper.ffiWrapper
                false -> FfiWrapper(wrapper) 
            }
            ffiWrapper.use {
                ffiBuilderConfig.addWrapper(ffiUri, it)
            }
        }
    }

    /**
     * Removes a wrapper with the specified URI key from the current configuration.
     *
     * @param uri The URI key of the wrapper to remove.
     * @return This [BaseConfigBuilder] instance for chaining calls.
     */
    fun removeWrapper(uri: String) {
        ffiUriFromString(uri).use { ffiUri ->
            ffiBuilderConfig.removeWrapper(ffiUri)
        }
    }

    fun addPackage(uri: String, wrapPackage: WrapPackage) {
        ffiUriFromString(uri).use { ffiUri ->
            val ffiWrapPackage = when (wrapPackage is WasmPackage) {
                true -> wrapPackage.ffiWrapPackage
                false -> FfiWrapPackage(wrapPackage)
            }
            ffiWrapPackage.use {
                ffiBuilderConfig.addPackage(ffiUri, it)
            }
        }
    }

    fun removePackage(uri: String) {
        ffiUriFromString(uri).use { ffiUri ->
            ffiBuilderConfig.removePackage(ffiUri)
        }
    }

    fun addRedirect(from: String, to: String) {
        ffiUriFromString(from).use { fromUri ->
            ffiUriFromString(to).use { toUri ->
                ffiBuilderConfig.addRedirect(fromUri, toUri)
            }
        }
    }

    fun removeRedirect(from: String) {
        ffiUriFromString(from).use { ffiUri ->
            ffiBuilderConfig.removeRedirect(ffiUri)
        }
    }

    fun addBundle(bundle: NativeBundle) {
        when (bundle) {
            NativeBundle.System -> ffiBuilderConfig.addSystemDefaults()
            NativeBundle.Web3 -> ffiBuilderConfig.addWeb3Defaults()
        }
    }

    /**
     * Returns a configured [FfiClient] instance.
     */
    fun build(): FfiClient = ffiBuilderConfig.build()

    override fun close() = ffiBuilderConfig.close()
}
