package io.polyglotted.aws.message;

import com.amazonaws.AbortedException;
import com.amazonaws.services.sqs.AmazonSQS;
import com.amazonaws.services.sqs.model.DeleteMessageRequest;
import com.amazonaws.services.sqs.model.Message;
import com.amazonaws.services.sqs.model.ReceiveMessageRequest;
import io.polyglotted.aws.config.AwsConfig;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import static io.polyglotted.aws.common.AwsClientFactory.createSqsClient;
import static io.polyglotted.aws.message.SubscriberUtil.safeInvoke;
import static io.polyglotted.aws.message.SubscriberUtil.stopThreadPool;
import static io.polyglotted.common.util.CollUtil.transformList;
import static java.lang.Runtime.getRuntime;
import static java.lang.Thread.interrupted;
import static java.util.concurrent.Executors.newFixedThreadPool;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

@Slf4j
@RequiredArgsConstructor
public class SqsSubscriber implements AutoCloseable {
    private final ExecutorService receiverPool = newSingleThreadExecutor();
    private final ExecutorService handlerPool = newFixedThreadPool(getRuntime().availableProcessors());
    private final AmazonSQS sqsClient;
    private final QueueConfig subscriber;
    private final Object messageHandler;

    public SqsSubscriber(AwsConfig config, QueueConfig subscriber, Object messageHandler) {
        this(createSqsClient(config), subscriber, messageHandler);
        receiverPool.execute(this::receiveMessage);
    }

    @Override
    public void close() {
        stopThreadPool(receiverPool, subscriber.getShutdownTimeInMillis(), TimeUnit.MILLISECONDS);
        stopThreadPool(handlerPool, subscriber.getShutdownTimeInMillis(), TimeUnit.MILLISECONDS);
    }

    private void receiveMessage() {
        if (!subscriber.isEnabled()) return;
        final String queueUrl = sqsClient.getQueueUrl(subscriber.getQueue()).getQueueUrl();
        while (!interrupted()) {
            try {
                ReceiveMessageRequest receiveMessageRequest = new ReceiveMessageRequest(queueUrl)
                    .withWaitTimeSeconds(subscriber.getWaitTimeInSeconds()).withMaxNumberOfMessages(subscriber.getMaxMessages());
                List<Message> messages = sqsClient.receiveMessage(receiveMessageRequest).getMessages();
                if (messages.size() > 0) { handleMessages(queueUrl, messages); }

            } catch (AbortedException ignored) {
                break;
            } catch (Exception ex) { log.error("error receiving message from sqs", ex); }
        }
    }

    private void handleMessages(String queueUrl, List<Message> messages) {
        for (Message message : messages) {
            handlerPool.submit(() -> {
                try {
                    safeInvoke(messageHandler, "handleMessage", new Class[]{String.class}, message.getBody());
                    sqsClient.deleteMessage(new DeleteMessageRequest(queueUrl, message.getReceiptHandle()));
                } catch (Exception ex) { log.error("error processing messages: " + transformList(messages, Message::getMessageId), ex); }
            });
        }
    }
}