package io.polyglotted.aws.config;

import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.EC2ContainerCredentialsProviderWrapper;
import com.amazonaws.auth.EnvironmentVariableCredentialsProvider;
import com.amazonaws.auth.STSAssumeRoleSessionCredentialsProvider;
import com.amazonaws.auth.STSSessionCredentialsProvider;
import com.amazonaws.auth.SystemPropertiesCredentialsProvider;
import com.amazonaws.auth.profile.ProfileCredentialsProvider;
import io.polyglotted.common.util.MapBuilder.ImmutableMapBuilder;

import java.util.Map;

import static com.amazonaws.services.securitytoken.AWSSecurityTokenServiceClientBuilder.standard;
import static io.polyglotted.common.util.MapBuilder.immutableMapBuilder;

@SuppressWarnings("unused")
public enum CredsProvider {
    CONFIG {
        @Override AWSCredentialsProvider provider(AwsConfig config) { return new ConfigCredsProvider(config); }
    },
    ENVIRONMENT {
        @Override AWSCredentialsProvider provider(AwsConfig config) { return new EnvironmentVariableCredentialsProvider(); }
    },
    SYSTEM_PROPERTIES {
        @Override AWSCredentialsProvider provider(AwsConfig config) { return new SystemPropertiesCredentialsProvider(); }
    },
    PROFILE {
        @Override AWSCredentialsProvider provider(AwsConfig config) { return new ProfileCredentialsProvider(); }
    },
    CONTAINER {
        @Override AWSCredentialsProvider provider(AwsConfig config) { return new EC2ContainerCredentialsProviderWrapper(); }
    },
    STS_ROLE {
        @Override AWSCredentialsProvider provider(AwsConfig config) {
            return new STSAssumeRoleSessionCredentialsProvider.Builder(config.getRoleArn(), config.getRoleSessionName())
                .withStsClient(standard().build()).build();
        }
    },
    STS_ROLE_PROXY {
        @Override AWSCredentialsProvider provider(AwsConfig config) {
            return new STSAssumeRoleSessionCredentialsProvider.Builder(config.getRoleArn(), config.getRoleSessionName())
                .withStsClient(standard().withCredentials(new ConfigCredsProvider(config)).build()).build();
        }
    },
    STS_SESSION {
        @Override AWSCredentialsProvider provider(AwsConfig config) {
            return new STSSessionCredentialsProvider(new EnvironmentVariableCredentialsProvider());
        }
    },
    STS_SESSION_PROXY {
        @Override AWSCredentialsProvider provider(AwsConfig config) {
            return new STSSessionCredentialsProvider(new ConfigCredsProvider(config));
        }
    };

    private static final Map<String, CredsProvider> INSTANCES;

    static {
        ImmutableMapBuilder<String, CredsProvider> builder = immutableMapBuilder();
        for (CredsProvider value : values()) { builder.put(value.name(), value).put(value.name().toLowerCase(), value); }
        INSTANCES = builder.build();
    }

    abstract AWSCredentialsProvider provider(AwsConfig config);

    public static AWSCredentialsProvider getProvider(AwsConfig config) { return INSTANCES.get(config.getProvider()).provider(config); }
}