/*
 * Decompiled with CFR 0.152.
 */
package io.perfana.client;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import io.perfana.client.PerfanaUtils;
import io.perfana.client.api.PerfanaCaller;
import io.perfana.client.api.PerfanaClientLogger;
import io.perfana.client.api.PerfanaConnectionSettings;
import io.perfana.client.api.TestContext;
import io.perfana.client.domain.Benchmark;
import io.perfana.client.domain.Message;
import io.perfana.client.domain.PerfanaEvent;
import io.perfana.client.domain.PerfanaMessage;
import io.perfana.client.domain.PerfanaTest;
import io.perfana.client.domain.Result;
import io.perfana.client.domain.Variable;
import io.perfana.client.exception.PerfanaAssertionsAreFalse;
import io.perfana.client.exception.PerfanaClientException;
import io.perfana.client.exception.PerfanaClientRuntimeException;
import io.perfana.eventscheduler.exception.handler.AbortSchedulerException;
import io.perfana.eventscheduler.exception.handler.KillSwitchException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.jetbrains.annotations.NotNull;

public final class PerfanaClient
implements PerfanaCaller {
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private final OkHttpClient client = new OkHttpClient();
    private final PerfanaClientLogger logger;
    private final TestContext context;
    private final PerfanaConnectionSettings settings;
    private final boolean assertResultsEnabled;
    private static final ObjectReader perfanaBenchmarkReader;
    private static final ObjectReader messageReader;
    private static final ObjectReader perfanaTestReader;
    private static final ObjectWriter perfanaMessageWriter;
    private static final ObjectWriter perfanaEventWriter;

    PerfanaClient(TestContext context, PerfanaConnectionSettings settings, boolean assertResultsEnabled, PerfanaClientLogger logger) {
        this.context = context;
        this.settings = settings;
        this.assertResultsEnabled = assertResultsEnabled;
        this.logger = logger;
    }

    @Override
    public void callPerfanaTestEndpoint(TestContext context, boolean completed) throws KillSwitchException {
        this.callPerfanaTestEndpoint(context, completed, Collections.emptyMap());
    }

    @Override
    public void callPerfanaTestEndpoint(TestContext context, boolean completed, Map<String, String> extraVariables) throws KillSwitchException {
        String json = PerfanaClient.perfanaMessageToJson(context, completed, extraVariables);
        Request request = this.createRequest("/api/test", json);
        try (Response result = this.client.newCall(request).execute();){
            this.logger.debug("test endpoint result: " + result);
            int responseCode = result.code();
            ResponseBody responseBody = result.body();
            if (responseCode == 401) {
                throw new AbortSchedulerException("Abort due to: not authorized (401) for [" + request + "]");
            }
            if (responseCode == 400) {
                if (responseBody != null) {
                    Message message = (Message)messageReader.readValue(responseBody.string());
                    throw new AbortSchedulerException(message.getMessage());
                }
                this.logger.error("No response body in test endpoint result: " + result);
            } else if (responseBody != null) {
                PerfanaTest test;
                if (!completed && (test = (PerfanaTest)perfanaTestReader.readValue(responseBody.string())).isAbort()) {
                    String message = test.getAbortMessage();
                    this.logger.info("abort requested by Perfana! Reason: '" + message + "'");
                    throw new KillSwitchException(message);
                }
            } else {
                this.logger.error("No response body in test endpoint result: " + result);
            }
        }
        catch (IOException e) {
            this.logger.error("failed to call Perfana test endpoint: " + e.getMessage());
        }
    }

    @NotNull
    private Request createRequest(String endPoint) {
        return this.createRequest(endPoint, null);
    }

    private Request createRequest(@NotNull String endpoint, String json) {
        this.logger.debug("call to endpoint: " + endpoint + (json != null ? " with json: " + json : ""));
        String url = PerfanaUtils.addSlashIfNeeded(this.settings.getPerfanaUrl(), endpoint);
        Request.Builder requestBuilder = new Request.Builder().url(url);
        if (json == null) {
            requestBuilder.get();
        } else {
            RequestBody body = RequestBody.create((String)json, (MediaType)JSON);
            requestBuilder.post(body);
        }
        if (this.settings.getApiKey() != null) {
            requestBuilder.addHeader("Authorization", "Bearer " + this.settings.getApiKey());
        }
        return requestBuilder.build();
    }

    @Override
    public void callPerfanaEvent(TestContext context, String eventTitle, String eventDescription) {
        this.logger.info("add Perfana event: " + eventDescription);
        String json = this.perfanaEventToJson(context, eventTitle, eventDescription);
        try {
            String result = this.post("/api/events", json);
            this.logger.debug("result: " + result);
        }
        catch (IOException e) {
            this.logger.error("failed to call Perfana event endpoint: " + e.getMessage());
        }
    }

    private String post(String endpoint, String json) throws IOException {
        Request request = this.createRequest(endpoint, json);
        try (Response response = this.client.newCall(request).execute();){
            ResponseBody responseBody = response.body();
            int responseCode = response.code();
            if (responseCode == 401) {
                this.logger.warn("ignoring: not authorised (401) to post to [" + endpoint + "]");
            }
            if (!response.isSuccessful()) {
                this.logger.warn("POST was not successful: " + response + " for request: " + request + " and body: " + json);
            }
            String string = responseBody == null ? "null" : responseBody.string();
            return string;
        }
    }

    public static String perfanaMessageToJson(TestContext context, boolean completed, Map<String, String> extraVariables) {
        PerfanaMessage.PerfanaMessageBuilder perfanaMessageBuilder = PerfanaMessage.builder().testRunId(context.getTestRunId()).workload(context.getWorkload()).testEnvironment(context.getTestEnvironment()).systemUnderTest(context.getSystemUnderTest()).version(context.getVersion()).cibuildResultsUrl(context.getCIBuildResultsUrl()).rampUp(String.valueOf(context.getRampupTime().getSeconds())).duration(String.valueOf(context.getPlannedDuration().getSeconds())).completed(completed).annotations(context.getAnnotations()).tags(context.getTags());
        context.getVariables().forEach((k, v) -> perfanaMessageBuilder.variable(Variable.builder().placeholder((String)k).value((String)v).build()));
        extraVariables.forEach((k, v) -> perfanaMessageBuilder.variable(Variable.builder().placeholder((String)k).value((String)v).build()));
        PerfanaMessage perfanaMessage = perfanaMessageBuilder.build();
        try {
            return perfanaMessageWriter.writeValueAsString((Object)perfanaMessage);
        }
        catch (JsonProcessingException e) {
            throw new PerfanaClientRuntimeException("Failed to write PerfanaMessage to json: " + perfanaMessage, (Exception)((Object)e));
        }
    }

    private String perfanaEventToJson(TestContext context, String eventTitle, String eventDescription) {
        PerfanaEvent event = PerfanaEvent.builder().systemUnderTest(context.getSystemUnderTest()).testEnvironment(context.getTestEnvironment()).title(eventTitle).description(eventDescription).tag(context.getWorkload()).build();
        try {
            return perfanaEventWriter.writeValueAsString((Object)event);
        }
        catch (JsonProcessingException e) {
            throw new PerfanaClientRuntimeException("Unable to transform PerfanaEvent to json", (Exception)((Object)e));
        }
    }

    private String callCheckAsserts() throws PerfanaClientException {
        String endPoint;
        try {
            endPoint = String.join((CharSequence)"/", "/api", "benchmark-results", this.encodeForURL(this.context.getSystemUnderTest()), this.encodeForURL(this.context.getTestRunId()));
        }
        catch (UnsupportedEncodingException e) {
            throw new PerfanaClientException("cannot encode Perfana url.", e);
        }
        Request request = this.createRequest(endPoint);
        int maxRetryCount = this.settings.getRetryMaxCount();
        long sleepDurationMillis = this.settings.getRetryDuration().toMillis();
        int retryCount = 0;
        String assertions = null;
        boolean assertionsAvailable = false;
        boolean checksSpecified = false;
        while (!assertionsAvailable && retryCount++ < maxRetryCount) {
            block25: {
                try (Response response = this.client.newCall(request).execute();){
                    ResponseBody responseBody = response.body();
                    int responseCode = response.code();
                    if (responseCode == 200) {
                        String string = assertions = responseBody == null ? "null" : responseBody.string();
                        if (assertions.contains("<!DOCTYPE html>")) {
                            throw new PerfanaClientException("Got html instead of json response for [" + endPoint + "]: [" + assertions + "]");
                        }
                        assertionsAvailable = true;
                        checksSpecified = true;
                        break block25;
                    }
                    if (responseCode == 204) {
                        assertionsAvailable = true;
                        checksSpecified = false;
                        break block25;
                    }
                    if (responseCode == 400) {
                        throw new PerfanaClientException("Something went wrong while evaluating the test run [" + this.context.getTestRunId() + "]");
                    }
                    if (responseCode == 404) {
                        throw new PerfanaClientException("Test run not found [" + this.context.getTestRunId() + "]");
                    }
                    if (responseCode == 202) {
                        this.logger.info(String.format("Trying to get test run check results at %s, attempt [%d/%d]. Returncode [%d]: Test run evaluation in progress ...", endPoint, retryCount, maxRetryCount, responseCode));
                        break block25;
                    }
                    if (responseCode == 401) {
                        throw new PerfanaClientException("Not authorized (401) for [" + endPoint + "]");
                    }
                    throw new PerfanaClientException("No action defined for dealing with http code (" + responseCode + ") for [" + endPoint + "]");
                }
                catch (IOException e) {
                    throw new PerfanaClientException("Exception while trying to get test run check results at [" + endPoint + "]", e);
                }
            }
            if (assertionsAvailable) continue;
            this.sleep(sleepDurationMillis);
        }
        if (!assertionsAvailable) {
            this.logger.warn("Failed to get test run check results at [" + endPoint + "], maximum attempts reached!");
            throw new PerfanaClientException("Failed to get test run check results at [" + endPoint + "], maximum attempts reached!");
        }
        return checksSpecified ? assertions : null;
    }

    private void sleep(long sleepDurationMillis) {
        try {
            Thread.sleep(sleepDurationMillis);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private String encodeForURL(String testRunId) throws UnsupportedEncodingException {
        return URLEncoder.encode(testRunId, "UTF-8").replaceAll("\\+", "%20");
    }

    public String assertResults() throws PerfanaClientException, PerfanaAssertionsAreFalse {
        Benchmark benchmark;
        if (!this.assertResultsEnabled) {
            String message = "Perfana assert results is not enabled and will not be checked.";
            this.logger.info(message);
            return message;
        }
        String assertions = this.callCheckAsserts();
        if (assertions == null) {
            return "No checks have been specified for this test run! Set assertResults property to false or create checks for key metrics";
        }
        try {
            benchmark = (Benchmark)perfanaBenchmarkReader.readValue(assertions);
        }
        catch (IOException e) {
            throw new PerfanaClientRuntimeException("Unable to parse benchmark message: " + assertions, e);
        }
        Optional<Result> baseline = Optional.ofNullable(benchmark.getBenchmarkBaselineTestRun());
        Optional<Result> previous = Optional.ofNullable(benchmark.getBenchmarkPreviousTestRun());
        Optional<Result> requirements = Optional.ofNullable(benchmark.getRequirements());
        requirements.ifPresent(r -> this.logger.info("Requirements: " + r.isResult()));
        baseline.ifPresent(r -> this.logger.info("Compared to baseline test run: " + r.isResult()));
        previous.ifPresent(r -> this.logger.info("Compared to previous test run: " + r.isResult()));
        StringBuilder text = new StringBuilder();
        if (assertions.contains("false")) {
            text.append("One or more Perfana assertions are failing: \n");
            requirements.filter(r -> !r.isResult()).ifPresent(r -> text.append("Requirements check failed: ").append(r.getDeeplink()).append("\n"));
            baseline.filter(r -> !r.isResult()).ifPresent(r -> text.append("Comparison check to baseline test run failed: ").append(r.getDeeplink()).append("\n"));
            previous.filter(r -> !r.isResult()).ifPresent(r -> text.append("Comparison check to previous test run failed: ").append(r.getDeeplink()).append("\n"));
            this.logger.info("Test run has failed checks: " + text);
            throw new PerfanaAssertionsAreFalse(text.toString());
        }
        text.append("All configured checks are OK: \n");
        requirements.ifPresent(r -> text.append(r.getDeeplink()).append("\n"));
        baseline.ifPresent(r -> text.append(r.getDeeplink()).append("\n"));
        previous.ifPresent(r -> text.append(r.getDeeplink()));
        return text.toString();
    }

    public String toString() {
        return "PerfanaClient [testRunId:" + this.context.getTestRunId() + " workload: " + this.context.getWorkload() + " testEnvironment: " + this.context.getTestEnvironment() + " Perfana url: " + this.settings.getPerfanaUrl() + "]";
    }

    static {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        perfanaBenchmarkReader = objectMapper.reader().forType(Benchmark.class);
        messageReader = objectMapper.reader().forType(Message.class);
        perfanaTestReader = objectMapper.reader().forType(PerfanaTest.class);
        perfanaMessageWriter = objectMapper.writer().forType(PerfanaMessage.class);
        perfanaEventWriter = objectMapper.writer().forType(PerfanaEvent.class);
    }
}

