/*
 * Decompiled with CFR 0.152.
 */
package io.perfana.client;

import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.ParseContext;
import com.jayway.jsonpath.Predicate;
import io.perfana.client.PerfanaExecutorEngine;
import io.perfana.client.api.PerfanaCaller;
import io.perfana.client.api.PerfanaClientLogger;
import io.perfana.client.api.PerfanaConnectionSettings;
import io.perfana.client.api.TestContext;
import io.perfana.client.exception.PerfanaAssertionsAreFalse;
import io.perfana.client.exception.PerfanaClientException;
import io.perfana.event.PerfanaEventBroadcaster;
import io.perfana.event.PerfanaEventProperties;
import io.perfana.event.ScheduleEvent;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;
import java.util.Map;
import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

public final class PerfanaClient
implements PerfanaCaller {
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private final OkHttpClient client = new OkHttpClient();
    private final PerfanaClientLogger logger;
    private final TestContext context;
    private final PerfanaConnectionSettings settings;
    private final boolean assertResultsEnabled;
    private final PerfanaEventBroadcaster broadcaster;
    private final PerfanaEventProperties eventProperties;
    private final List<ScheduleEvent> scheduleEvents;
    private PerfanaExecutorEngine executorEngine;
    private boolean isSessionStopped = false;

    PerfanaClient(TestContext context, PerfanaConnectionSettings settings, boolean assertResultsEnabled, PerfanaEventBroadcaster broadcaster, PerfanaEventProperties eventProperties, List<ScheduleEvent> scheduleEvents, PerfanaClientLogger logger) {
        this.context = context;
        this.settings = settings;
        this.assertResultsEnabled = assertResultsEnabled;
        this.eventProperties = eventProperties;
        this.broadcaster = broadcaster;
        this.scheduleEvents = scheduleEvents;
        this.logger = logger;
    }

    public void startSession() {
        this.logger.info("Perfana start session");
        this.isSessionStopped = false;
        this.executorEngine = new PerfanaExecutorEngine(this.logger);
        this.broadcaster.broadcastBeforeTest(this.context, this.eventProperties);
        this.executorEngine.startKeepAliveThread(this, this.context, this.settings, this.broadcaster, this.eventProperties);
        this.executorEngine.startCustomEventScheduler(this, this.context, this.scheduleEvents, this.broadcaster, this.eventProperties);
        this.callPerfanaEvent(this.context, "Test start");
    }

    public void stopSession() throws PerfanaClientException, PerfanaAssertionsAreFalse {
        this.logger.info("Perfana end session.");
        this.isSessionStopped = true;
        this.executorEngine.shutdownThreadsNow();
        this.callPerfanaEvent(this.context, "Test finish");
        this.broadcaster.broadcastAfterTest(this.context, this.eventProperties);
        this.callPerfanaTestEndpoint(this.context, true);
        String text = this.assertResults();
        this.logger.info(String.format("the assertion text: %s", text));
    }

    public boolean isSessionStopped() {
        return this.isSessionStopped;
    }

    public void abortSession() {
        this.logger.warn("Perfana session abort called.");
        this.isSessionStopped = true;
        this.executorEngine.shutdownThreadsNow();
        this.callPerfanaEvent(this.context, "Test aborted");
        this.broadcaster.broadcastAfterTest(this.context, this.eventProperties);
    }

    @Override
    public void callPerfanaTestEndpoint(TestContext context, boolean completed) {
        String json = PerfanaClient.perfanaMessageToJson(context, completed);
        String testUrl = this.settings.getPerfanaUrl() + "/test";
        this.logger.debug(String.format("call to endpoint: %s with json: %s", testUrl, json));
        try {
            String result = this.post(testUrl, json);
            this.logger.debug("result: " + result);
        }
        catch (IOException e) {
            this.logger.error("failed to call perfana: " + e.getMessage());
        }
    }

    @Override
    public void callPerfanaEvent(TestContext context, String eventDescription) {
        this.logger.info("add Perfana event: " + eventDescription);
        String json = this.perfanaEventToJson(context, eventDescription);
        String eventsUrl = this.settings.getPerfanaUrl() + "/events";
        this.logger.debug(String.format("add perfana event to endpoint: %s with json: %s", eventsUrl, json));
        try {
            String result = this.post(eventsUrl, json);
            this.logger.debug("result: " + result);
        }
        catch (IOException e) {
            this.logger.error("failed to call perfana: " + e.getMessage());
        }
    }

    private String post(String url, String json) throws IOException {
        RequestBody body = RequestBody.create((MediaType)JSON, (String)json);
        Request request = new Request.Builder().url(url).post(body).build();
        try (Response response = this.client.newCall(request).execute();){
            ResponseBody responseBody = response.body();
            if (!response.isSuccessful()) {
                this.logger.warn(String.format("POST was not successful: %s for request: %s and body: %s", response, request, json));
            }
            String string = responseBody == null ? "null" : responseBody.string();
            return string;
        }
    }

    public static String perfanaMessageToJson(TestContext context, boolean completed) {
        String annotations;
        JSONObject json = new JSONObject();
        Map<String, String> variables = context.getVariables();
        if (variables != null && !variables.isEmpty()) {
            JSONArray variablesArray = new JSONArray();
            variables.forEach((key, value) -> variablesArray.add((Object)PerfanaClient.createVariables(key, value)));
            json.put((Object)"variables", (Object)variablesArray);
        }
        if ((annotations = context.getAnnotations()) != null && !annotations.isEmpty()) {
            json.put((Object)"annotations", (Object)annotations);
        }
        json.put((Object)"testRunId", (Object)context.getTestRunId());
        json.put((Object)"testType", (Object)context.getTestType());
        json.put((Object)"testEnvironment", (Object)context.getTestEnvironment());
        json.put((Object)"application", (Object)context.getApplication());
        json.put((Object)"applicationRelease", (Object)context.getApplicationRelease());
        json.put((Object)"CIBuildResultsUrl", (Object)context.getCIBuildResultsUrl());
        json.put((Object)"rampUp", (Object)String.valueOf(context.getRampupTime().getSeconds()));
        json.put((Object)"duration", (Object)String.valueOf(context.getPlannedDuration().getSeconds()));
        json.put((Object)"completed", (Object)completed);
        return json.toJSONString();
    }

    private static JSONObject createVariables(String name, String value) {
        JSONObject variables = new JSONObject();
        variables.put((Object)"placeholder", (Object)name);
        variables.put((Object)"value", (Object)value);
        return variables;
    }

    private String perfanaEventToJson(TestContext context, String eventDescription) {
        JSONObject json = new JSONObject();
        json.put((Object)"application", (Object)context.getApplication());
        json.put((Object)"testEnvironment", (Object)context.getTestEnvironment());
        json.put((Object)"title", (Object)context.getTestRunId());
        json.put((Object)"description", (Object)eventDescription);
        JSONArray tags = new JSONArray();
        tags.add((Object)context.getTestType());
        json.put((Object)"tags", (Object)tags);
        return json.toJSONString();
    }

    private String callCheckAsserts() throws PerfanaClientException {
        String url;
        try {
            url = String.join((CharSequence)"/", this.settings.getPerfanaUrl(), "get-benchmark-results", this.encodeForURL(this.context.getApplication()), this.encodeForURL(this.context.getTestRunId()));
        }
        catch (UnsupportedEncodingException e) {
            throw new PerfanaClientException("cannot encode perfana url.", e);
        }
        Request request = new Request.Builder().url(url).get().build();
        int retryCount = 0;
        String assertions = null;
        boolean assertionsAvailable = false;
        while (retryCount++ < this.settings.getRetryMaxCount()) {
            try {
                Thread.sleep(this.settings.getRetryDuration().toMillis());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            try {
                Response response = this.client.newCall(request).execute();
                Throwable throwable = null;
                try {
                    ResponseBody responseBody = response.body();
                    if (response.code() == 200) {
                        assertions = responseBody == null ? "null" : responseBody.string();
                        assertionsAvailable = true;
                        break;
                    }
                    String message = responseBody == null ? response.message() : responseBody.string();
                    this.logger.info(String.format("failed to retrieve assertions for url [%s] code [%d] retry [%d/%d] %s", url, response.code(), retryCount, this.settings.getRetryMaxCount(), message));
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (response == null) continue;
                    if (throwable != null) {
                        try {
                            response.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    response.close();
                }
            }
            catch (IOException e) {
                throw new PerfanaClientException(String.format("unable to retrieve assertions for url [%s]", url), e);
            }
        }
        if (!assertionsAvailable) {
            this.logger.warn(String.format("failed to retrieve assertions for url [%s], no more retries left!", url));
            throw new PerfanaClientException(String.format("unable to retrieve assertions for url [%s]", url));
        }
        return assertions;
    }

    private String encodeForURL(String testRunId) throws UnsupportedEncodingException {
        return URLEncoder.encode(testRunId, "UTF-8").replaceAll("\\+", "%20");
    }

    private String assertResults() throws PerfanaClientException, PerfanaAssertionsAreFalse {
        if (!this.assertResultsEnabled) {
            String message = "Perfana assert results is not enabled and will not be checked.";
            this.logger.info(message);
            return message;
        }
        String assertions = this.callCheckAsserts();
        if (assertions == null) {
            throw new PerfanaClientException("Perfana assertions could not be checked, received null.");
        }
        Configuration config = Configuration.defaultConfiguration().addOptions(new Option[]{Option.SUPPRESS_EXCEPTIONS});
        ParseContext parseContext = JsonPath.using((Configuration)config);
        DocumentContext doc = parseContext.parse(assertions);
        Boolean benchmarkBaselineTestRunResult = (Boolean)doc.read("$.benchmarkBaselineTestRun.result", new Predicate[0]);
        String benchmarkBaselineTestRunDeeplink = (String)doc.read("$.benchmarkBaselineTestRun.deeplink", new Predicate[0]);
        Boolean benchmarkPreviousTestRunResult = (Boolean)doc.read("$.benchmarkPreviousTestRun.result", new Predicate[0]);
        String benchmarkPreviousTestRunDeeplink = (String)doc.read("$.benchmarkPreviousTestRun.deeplink", new Predicate[0]);
        Boolean requirementsResult = (Boolean)doc.read("$.requirements.result", new Predicate[0]);
        String requirementsDeeplink = (String)doc.read("$.requirements.deeplink", new Predicate[0]);
        this.logger.info(String.format("benchmarkBaselineTestRunResult: %s", benchmarkBaselineTestRunResult));
        this.logger.info(String.format("benchmarkPreviousTestRunResult: %s", benchmarkPreviousTestRunResult));
        this.logger.info(String.format("requirementsResult: %s", requirementsResult));
        StringBuilder text = new StringBuilder();
        if (assertions.contains("false")) {
            text.append("One or more Perfana assertions are failing: \n");
            if (PerfanaClient.hasFailed(requirementsResult)) {
                text.append(String.format("Requirements failed: %s\n", requirementsDeeplink));
            }
            if (PerfanaClient.hasFailed(benchmarkPreviousTestRunResult)) {
                text.append(String.format("Benchmark to previous test run failed: %s\n", benchmarkPreviousTestRunDeeplink));
            }
            if (PerfanaClient.hasFailed(benchmarkBaselineTestRunResult)) {
                text.append(String.format("Benchmark to baseline test run failed: %s", benchmarkBaselineTestRunDeeplink));
            }
            this.logger.info(String.format("assertionText: %s", text));
            throw new PerfanaAssertionsAreFalse(text.toString());
        }
        text.append("All Perfana assertions are OK: \n");
        if (requirementsResult.booleanValue()) {
            text.append(requirementsDeeplink).append("\n");
        }
        if (PerfanaClient.hasSucceeded(benchmarkPreviousTestRunResult)) {
            text.append(benchmarkPreviousTestRunDeeplink).append("\n");
        }
        if (PerfanaClient.hasSucceeded(benchmarkBaselineTestRunResult)) {
            text.append(benchmarkBaselineTestRunDeeplink);
        }
        return text.toString();
    }

    private static boolean hasSucceeded(Boolean testResult) {
        return testResult != null && testResult != false;
    }

    private static boolean hasFailed(Boolean testResult) {
        return testResult != null && testResult == false;
    }

    public String toString() {
        return String.format("PerfanaClient [testRunId:%s testType:%s testEnv:%s perfanaUrl:%s]", this.context.getTestRunId(), this.context.getTestType(), this.context.getTestEnvironment(), this.settings.getPerfanaUrl());
    }
}

