/*
 * Decompiled with CFR 0.152.
 */
package io.perfana.client;

import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import com.jayway.jsonpath.ParseContext;
import com.jayway.jsonpath.Predicate;
import io.perfana.client.PerfanaClientException;
import io.perfana.event.PerfanaEventBroadcaster;
import io.perfana.event.PerfanaEventProperties;
import io.perfana.event.ScheduleEvent;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import net.minidev.json.JSONArray;
import net.minidev.json.JSONObject;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

public final class PerfanaClient {
    private static final MediaType JSON = MediaType.parse((String)"application/json; charset=utf-8");
    private final OkHttpClient client = new OkHttpClient();
    private Logger logger;
    private final String application;
    private final String testType;
    private final String testEnvironment;
    private final String testRunId;
    private final String CIBuildResultsUrl;
    private final String applicationRelease;
    private final String perfanaUrl;
    private final Duration rampupTime;
    private final Duration plannedDuration;
    private final String annotations;
    private final Map<String, String> variables;
    private final boolean assertResultsEnabled;
    private final int retryMaxCount;
    private final Duration retryDuration;
    private final Duration keepAliveDuration;
    private final PerfanaEventBroadcaster broadcaster;
    private final PerfanaEventProperties eventProperties;
    private final List<ScheduleEvent> scheduleEvents;
    private ScheduledExecutorService executorKeepAlive;
    private ScheduledExecutorService executorCustomEvents;

    PerfanaClient(String application, String testType, String testEnvironment, String testRunId, String CIBuildResultsUrl, String applicationRelease, Duration rampupTime, Duration constantLoadTime, String perfanaUrl, String annotations, Map<String, String> variables, boolean assertResultsEnabled, PerfanaEventBroadcaster broadcaster, PerfanaEventProperties eventProperties, int retryMaxCount, Duration retryDuration, Duration keepAliveDuration, List<ScheduleEvent> scheduleEvents) {
        this.application = application;
        this.testType = testType;
        this.testEnvironment = testEnvironment;
        this.testRunId = testRunId;
        this.CIBuildResultsUrl = CIBuildResultsUrl;
        this.applicationRelease = applicationRelease;
        this.rampupTime = rampupTime;
        this.plannedDuration = rampupTime.plus(constantLoadTime);
        this.perfanaUrl = perfanaUrl;
        this.annotations = annotations;
        this.variables = variables;
        this.assertResultsEnabled = assertResultsEnabled;
        this.broadcaster = broadcaster;
        this.eventProperties = eventProperties;
        this.retryMaxCount = retryMaxCount;
        this.retryDuration = retryDuration;
        this.keepAliveDuration = keepAliveDuration;
        this.scheduleEvents = scheduleEvents;
    }

    public void startSession() {
        this.logger.info("Perfana start session");
        this.logger.info("Perfana broadcast event before test");
        this.callPerfanaEvent("Test start");
        this.broadcaster.broadcastBeforeTest(this.testRunId, this.eventProperties);
        this.startCustomEventScheduler(this.scheduleEvents);
        if (this.executorKeepAlive != null) {
            throw new RuntimeException("Cannot start perfana session multiple times!");
        }
        this.logger.info(String.format("Calling Perfana (%s) keep alive every %d seconds.", this.perfanaUrl, this.keepAliveDuration.getSeconds()));
        this.executorKeepAlive = Executors.newSingleThreadScheduledExecutor();
        KeepAliveRunner keepAliveRunner = new KeepAliveRunner(this);
        this.executorKeepAlive.scheduleAtFixedRate(keepAliveRunner, 0L, this.keepAliveDuration.getSeconds(), TimeUnit.SECONDS);
    }

    private void startCustomEventScheduler(List<ScheduleEvent> scheduleEvents) {
        if (scheduleEvents != null && !scheduleEvents.isEmpty()) {
            this.logger.info(this.createEventScheduleMessage(scheduleEvents));
            this.executorCustomEvents = Executors.newScheduledThreadPool(4);
            scheduleEvents.forEach(event -> this.addToExecutor(this.executorCustomEvents, (ScheduleEvent)event));
        } else {
            this.logger.info("No custom Perfana schedule events found.");
        }
    }

    private String createEventScheduleMessage(List<ScheduleEvent> scheduleEvents) {
        StringBuilder message = new StringBuilder();
        message.append("=== custom Perfana events schedule ===");
        scheduleEvents.forEach(event -> message.append("\n==> ").append(event));
        return message.toString();
    }

    private void addToExecutor(ScheduledExecutorService executorService, ScheduleEvent event) {
        executorService.schedule(new EventRunner(this.testRunId, this.eventProperties, event, this.broadcaster, this), event.getDuration().getSeconds(), TimeUnit.SECONDS);
    }

    public void stopSession() throws PerfanaClientException {
        List<Runnable> runnables;
        this.logger.info("Perfana end session.");
        if (this.executorKeepAlive != null) {
            this.executorKeepAlive.shutdown();
        }
        if (this.executorCustomEvents != null && (runnables = this.executorCustomEvents.shutdownNow()).size() > 0) {
            this.logger.warn("There are " + runnables.size() + " custom Perfana events that are not (fully) executed!");
        }
        this.executorKeepAlive = null;
        this.executorCustomEvents = null;
        this.logger.info("Perfana broadcast event after test");
        this.callPerfanaEvent("Test finish");
        this.broadcaster.broadcastAfterTest(this.testRunId, this.eventProperties);
        this.callPerfana(true);
        this.assertResults();
    }

    void injectLogger(Logger logger) {
        this.logger = logger;
    }

    private void callPerfana(boolean completed) {
        String json = this.perfanaMessageToJson(this.application, this.testType, this.testEnvironment, this.testRunId, this.CIBuildResultsUrl, this.applicationRelease, this.rampupTime, this.plannedDuration, this.annotations, this.variables, completed);
        this.logger.debug(String.format("Call to endpoint: %s with json: %s", this.perfanaUrl, json));
        try {
            String result = this.post(this.perfanaUrl + "/test", json);
            this.logger.debug("Result: " + result);
        }
        catch (IOException e) {
            this.logger.error("Failed to call perfana: " + e.getMessage());
        }
    }

    private void callPerfanaEvent(String eventDescription) {
        String json = this.perfanaEventToJson(this.application, this.testType, this.testEnvironment, this.testRunId, eventDescription);
        this.logger.debug(String.format("Call to endpoint: %s with json: %s", this.perfanaUrl, json));
        try {
            String result = this.post(this.perfanaUrl + "/events", json);
            this.logger.debug("Result: " + result);
        }
        catch (IOException e) {
            this.logger.error("Failed to call perfana: " + e.getMessage());
        }
    }

    private String post(String url, String json) throws IOException {
        RequestBody body = RequestBody.create((MediaType)JSON, (String)json);
        Request request = new Request.Builder().url(url).post(body).build();
        try (Response response = this.client.newCall(request).execute();){
            ResponseBody responseBody = response.body();
            String string = responseBody == null ? "null" : responseBody.string();
            return string;
        }
    }

    private String perfanaMessageToJson(String application, String testType, String testEnvironment, String testRunId, String CIBuildResultsUrl, String applicationRelease, Duration rampupTime, Duration plannedDuration, String annotations, Map<String, String> variables, boolean completed) {
        JSONObject json = new JSONObject();
        if (variables != null && !variables.isEmpty()) {
            JSONArray variablesArray = new JSONArray();
            variables.forEach((key, value) -> variablesArray.add((Object)PerfanaClient.createVariables(key, value)));
            json.put((Object)"variables", (Object)variablesArray);
        }
        if (annotations != null && annotations.isEmpty()) {
            json.put((Object)"annotations", (Object)annotations);
        }
        json.put((Object)"testRunId", (Object)testRunId);
        json.put((Object)"testType", (Object)testType);
        json.put((Object)"testEnvironment", (Object)testEnvironment);
        json.put((Object)"application", (Object)application);
        json.put((Object)"applicationRelease", (Object)applicationRelease);
        json.put((Object)"CIBuildResultsUrl", (Object)CIBuildResultsUrl);
        json.put((Object)"rampUp", (Object)String.valueOf(rampupTime.getSeconds()));
        json.put((Object)"duration", (Object)String.valueOf(plannedDuration.getSeconds()));
        json.put((Object)"completed", (Object)completed);
        return json.toJSONString();
    }

    private static JSONObject createVariables(String name, String value) {
        JSONObject variables = new JSONObject();
        variables.put((Object)"placeholder", (Object)name);
        variables.put((Object)"value", (Object)value);
        return variables;
    }

    private String perfanaEventToJson(String application, String testType, String testEnvironment, String testRunId, String eventDescription) {
        JSONObject json = PerfanaClient.createVariables(application, testEnvironment);
        json.put((Object)"testEnvironment", (Object)testEnvironment);
        json.put((Object)"title", (Object)testRunId);
        json.put((Object)"description", (Object)eventDescription);
        JSONArray tags = new JSONArray();
        tags.add((Object)testType);
        json.put((Object)"tags", (Object)tags);
        return json.toJSONString();
    }

    private String callCheckAsserts() throws PerfanaClientException {
        String url;
        try {
            url = String.join((CharSequence)"/", this.perfanaUrl, "get-benchmark-results", URLEncoder.encode(this.application, "UTF-8").replaceAll("\\+", "%20"), URLEncoder.encode(this.testRunId, "UTF-8").replaceAll("\\+", "%20"));
        }
        catch (UnsupportedEncodingException e) {
            throw new PerfanaClientException("Cannot encode perfana url.", e);
        }
        Request request = new Request.Builder().url(url).get().build();
        int retryCount = 0;
        String assertions = null;
        boolean assertionsAvailable = false;
        while (retryCount++ < this.retryMaxCount) {
            try {
                Thread.sleep(this.retryDuration.toMillis());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            try {
                Response response = this.client.newCall(request).execute();
                Throwable throwable = null;
                try {
                    ResponseBody responseBody = response.body();
                    if (response.code() == 200) {
                        assertions = responseBody == null ? "null" : responseBody.string();
                        assertionsAvailable = true;
                        break;
                    }
                    String message = responseBody == null ? response.message() : responseBody.string();
                    this.logger.info(String.format("failed to retrieve assertions for url [%s] code [%d] retry [%d/%d] %s", url, response.code(), retryCount, this.retryMaxCount, message));
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (response == null) continue;
                    if (throwable != null) {
                        try {
                            response.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    response.close();
                }
            }
            catch (IOException e) {
                throw new PerfanaClientException(String.format("Unable to retrieve assertions for url [%s]", url), e);
            }
        }
        if (!assertionsAvailable) {
            this.logger.warn(String.format("Failed to retrieve assertions for url [%s], no more retries left!", url));
            throw new PerfanaClientException(String.format("Unable to retrieve assertions for url [%s]", url));
        }
        return assertions;
    }

    private String assertResults() throws PerfanaClientException {
        if (!this.assertResultsEnabled) {
            String message = "Perfana assert results is not enabled.";
            this.logger.info(message);
            return message;
        }
        String assertions = this.callCheckAsserts();
        if (assertions == null) {
            throw new PerfanaClientException("Perfana assertions could not be checked, received null.");
        }
        Configuration config = Configuration.defaultConfiguration().addOptions(new Option[]{Option.SUPPRESS_EXCEPTIONS});
        ParseContext parseContext = JsonPath.using((Configuration)config);
        DocumentContext documentContext = parseContext.parse(assertions);
        Boolean benchmarkBaselineTestRunResult = (Boolean)documentContext.read("$.benchmarkBaselineTestRun.result", new Predicate[0]);
        String benchmarkBaselineTestRunDeeplink = (String)documentContext.read("$.benchmarkBaselineTestRun.deeplink", new Predicate[0]);
        Boolean benchmarkPreviousTestRunResult = (Boolean)documentContext.read("$.benchmarkPreviousTestRun.result", new Predicate[0]);
        String benchmarkPreviousTestRunDeeplink = (String)documentContext.read("$.benchmarkPreviousTestRun.deeplink", new Predicate[0]);
        Boolean requirementsResult = (Boolean)documentContext.read("$.requirements.result", new Predicate[0]);
        String requirementsDeeplink = (String)documentContext.read("$.requirements.deeplink", new Predicate[0]);
        this.logger.info(String.format("benchmarkBaselineTestRunResult: %s", benchmarkBaselineTestRunResult));
        this.logger.info(String.format("benchmarkPreviousTestRunResult: %s", benchmarkPreviousTestRunResult));
        this.logger.info(String.format("requirementsResult: %s", requirementsResult));
        StringBuilder text = new StringBuilder();
        if (assertions.contains("false")) {
            text.append("One or more Perfana assertions are failing: \n");
            if (requirementsResult != null && !requirementsResult.booleanValue()) {
                text.append(String.format("Requirements failed: %s\n", requirementsDeeplink));
            }
            if (benchmarkPreviousTestRunResult != null && !benchmarkPreviousTestRunResult.booleanValue()) {
                text.append(String.format("Benchmark to previous test run failed: %s\n", benchmarkPreviousTestRunDeeplink));
            }
            if (benchmarkBaselineTestRunResult != null && !benchmarkBaselineTestRunResult.booleanValue()) {
                text.append(String.format("Benchmark to baseline test run failed: %s", benchmarkBaselineTestRunDeeplink));
            }
            this.logger.info(String.format("assertionText: %s", text));
            throw new PerfanaClientException(text.toString());
        }
        text.append("All Perfana assertions are OK: \n");
        if (requirementsResult.booleanValue()) {
            text.append(requirementsDeeplink).append("\n");
        }
        if (benchmarkPreviousTestRunResult != null && benchmarkPreviousTestRunResult.booleanValue()) {
            text.append(benchmarkPreviousTestRunDeeplink).append("\n");
        }
        if (benchmarkBaselineTestRunResult != null && benchmarkBaselineTestRunResult.booleanValue()) {
            text.append(benchmarkBaselineTestRunDeeplink);
        }
        this.logger.info(String.format("The assertionText: %s", text));
        return text.toString();
    }

    public String toString() {
        return String.format("PerfanaClient [testRunId:%s testType:%s testEnv:%s perfanaUrl:%s]", this.testRunId, this.testType, this.testEnvironment, this.perfanaUrl);
    }

    public class EventRunner
    implements Runnable {
        private final String testId;
        private final PerfanaEventProperties eventProperties;
        private final ScheduleEvent event;
        private final PerfanaEventBroadcaster eventBroadcaster;
        private final PerfanaClient client;

        public EventRunner(String testId, PerfanaEventProperties eventProperties, ScheduleEvent event, PerfanaEventBroadcaster eventBroadcaster, PerfanaClient perfanaClient) {
            this.testId = testId;
            this.eventProperties = eventProperties;
            this.event = event;
            this.eventBroadcaster = eventBroadcaster;
            this.client = perfanaClient;
        }

        @Override
        public void run() {
            this.client.logger.info("Fire Perfana custom event [" + this.event + "]");
            this.client.callPerfanaEvent(this.event.getName());
            this.eventBroadcaster.broadcastCustomEvent(this.testId, this.eventProperties, this.event);
        }

        public String toString() {
            return String.format("EventRunner for event %s for testId %s", this.event, this.testId);
        }
    }

    public static interface Logger {
        public void info(String var1);

        public void warn(String var1);

        public void error(String var1);

        public void debug(String var1);
    }

    public static class KeepAliveRunner
    implements Runnable {
        private final PerfanaClient client;

        KeepAliveRunner(PerfanaClient client) {
            this.client = client;
        }

        @Override
        public void run() {
            this.client.callPerfana(false);
            this.client.broadcaster.broadCastKeepAlive(this.client.testRunId, this.client.eventProperties);
        }

        public String toString() {
            return "KeepAliveRunner for " + this.client;
        }
    }
}

