/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.container.web.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public abstract class AbstractFileServlet
extends HttpServlet {
    private static final long serialVersionUID = 1L;
    private static final Logger logger = Logger.getLogger(AbstractFileServlet.class.getName());
    private static final Long DEFAULT_EXPIRE_TIME_IN_SECONDS = TimeUnit.DAYS.toSeconds(30L);
    private static final long ONE_SECOND_IN_MILLIS = TimeUnit.SECONDS.toMillis(1L);
    private static final String ETAG = "W/\"%s-%s\"";
    private static final Pattern RANGE_PATTERN = Pattern.compile("^bytes=[0-9]*-[0-9]*(,[0-9]*-[0-9]*)*$");
    private static final String MULTIPART_BOUNDARY = UUID.randomUUID().toString();
    private static final String CONTENT_DISPOSITION_HEADER = "%s;filename=\"%2$s\"; filename*=UTF-8''%2$s";
    private static final String ERROR_UNSUPPORTED_ENCODING = "UTF-8 is apparently not supported on this platform.";
    private static final int DEFAULT_STREAM_BUFFER_SIZE = 10240;

    protected void doHead(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.doRequest(request, response, true);
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.doRequest(request, response, false);
    }

    private void doRequest(HttpServletRequest request, HttpServletResponse response, boolean head) throws IOException {
        Resource resource;
        try {
            resource = new Resource(this.getFile(request));
        }
        catch (RedirectException ex) {
            logger.log(Level.FINE, "Redirecting client to: " + ex.location);
            response.setHeader("Location", ex.location);
            response.sendError(302);
            return;
        }
        catch (IllegalArgumentException e) {
            logger.log(Level.FINE, "Got an IllegalArgumentException from user code; interpreting it as 400 Bad Request.", e);
            response.sendError(400);
            return;
        }
        if (resource.file == null) {
            this.handleFileNotFound(request, response);
            return;
        }
        if (this.preconditionFailed(request, resource)) {
            response.sendError(412);
            return;
        }
        this.setCacheHeaders(response, resource, this.getExpireTime(request, resource.file));
        if (this.notModified(request, resource)) {
            response.setStatus(304);
            return;
        }
        List<Range> ranges = this.getRanges(request, resource);
        if (ranges == null) {
            response.setHeader("Content-Range", "bytes */" + resource.length);
            response.sendError(416);
            return;
        }
        if (!ranges.isEmpty()) {
            response.setStatus(206);
        } else {
            ranges.add(new Range(0L, resource.length - 1L));
        }
        String contentType = this.setContentHeaders(request, response, resource, ranges);
        if (head) {
            return;
        }
        this.writeContent(response, resource, ranges, contentType);
    }

    protected abstract File getFile(HttpServletRequest var1) throws IllegalArgumentException, RedirectException;

    protected void handleFileNotFound(HttpServletRequest request, HttpServletResponse response) throws IOException {
        response.sendError(404);
    }

    protected long getExpireTime(HttpServletRequest request, File file) {
        return DEFAULT_EXPIRE_TIME_IN_SECONDS;
    }

    protected String getContentType(HttpServletRequest request, File file) {
        return AbstractFileServlet.coalesce(request.getServletContext().getMimeType(file.getName()), "application/octet-stream");
    }

    private boolean preconditionFailed(HttpServletRequest request, Resource resource) {
        String match = request.getHeader("If-Match");
        long unmodified = request.getDateHeader("If-Unmodified-Since");
        return match != null ? !AbstractFileServlet.matches(match, resource.eTag) : unmodified != -1L && AbstractFileServlet.modified(unmodified, resource.lastModified);
    }

    private void setCacheHeaders(HttpServletResponse response, Resource resource, long expires) {
        AbstractFileServlet.setCacheHeaders(response, expires);
        response.setHeader("ETag", resource.eTag);
        response.setDateHeader("Last-Modified", resource.lastModified);
    }

    private boolean notModified(HttpServletRequest request, Resource resource) {
        String noMatch = request.getHeader("If-None-Match");
        long modified = request.getDateHeader("If-Modified-Since");
        return noMatch != null ? AbstractFileServlet.matches(noMatch, resource.eTag) : modified != -1L && !AbstractFileServlet.modified(modified, resource.lastModified);
    }

    private List<Range> getRanges(HttpServletRequest request, Resource resource) {
        ArrayList<Range> ranges = new ArrayList<Range>(1);
        String rangeHeader = request.getHeader("Range");
        if (rangeHeader == null) {
            return ranges;
        }
        if (!RANGE_PATTERN.matcher(rangeHeader).matches()) {
            return null;
        }
        String ifRange = request.getHeader("If-Range");
        if (ifRange != null && !ifRange.equals(resource.eTag)) {
            try {
                long ifRangeTime = request.getDateHeader("If-Range");
                if (ifRangeTime != -1L && AbstractFileServlet.modified(ifRangeTime, resource.lastModified)) {
                    return ranges;
                }
            }
            catch (IllegalArgumentException ifRangeHeaderIsInvalid) {
                logger.log(Level.FINE, "If-Range header is invalid. Let's just return full file then.", ifRangeHeaderIsInvalid);
                return ranges;
            }
        }
        for (String rangeHeaderPart : rangeHeader.split("=")[1].split(",")) {
            Range range = this.parseRange(rangeHeaderPart, resource.length);
            if (range == null) {
                return null;
            }
            ranges.add(range);
        }
        return ranges;
    }

    private Range parseRange(String range, long length) {
        long start = AbstractFileServlet.sublong(range, 0, range.indexOf(45));
        long end = AbstractFileServlet.sublong(range, range.indexOf(45) + 1, range.length());
        if (start == -1L) {
            start = length - end;
            end = length - 1L;
        } else if (end == -1L || end > length - 1L) {
            end = length - 1L;
        }
        if (start > end) {
            return null;
        }
        return new Range(start, end);
    }

    protected String setContentHeaders(HttpServletRequest request, HttpServletResponse response, Resource resource, List<Range> ranges) {
        String contentType = this.getContentType(request, resource.file);
        response.setHeader("Accept-Ranges", "bytes");
        if (ranges.size() == 1) {
            Range range = ranges.get(0);
            response.setContentType(contentType);
            response.setHeader("Content-Length", String.valueOf(range.length));
            if (response.getStatus() == 206) {
                response.setHeader("Content-Range", "bytes " + range.start + "-" + range.end + "/" + resource.length);
            }
        } else {
            response.setContentType("multipart/byteranges; boundary=" + MULTIPART_BOUNDARY);
        }
        return contentType;
    }

    private void writeContent(HttpServletResponse response, Resource resource, List<Range> ranges, String contentType) throws IOException {
        ServletOutputStream output = response.getOutputStream();
        if (ranges.size() == 1) {
            Range range = ranges.get(0);
            AbstractFileServlet.stream(resource.file, (OutputStream)output, range.start, range.length);
        } else {
            for (Range range : ranges) {
                output.println();
                output.println("--" + MULTIPART_BOUNDARY);
                output.println("Content-Type: " + contentType);
                output.println("Content-Range: bytes " + range.start + "-" + range.end + "/" + resource.length);
                AbstractFileServlet.stream(resource.file, (OutputStream)output, range.start, range.length);
            }
            output.println();
            output.println("--" + MULTIPART_BOUNDARY + "--");
        }
    }

    private static boolean matches(String matchHeader, String eTag) {
        Object[] matchValues = matchHeader.split("\\s*,\\s*");
        Arrays.sort(matchValues);
        return Arrays.binarySearch(matchValues, eTag) > -1 || Arrays.binarySearch(matchValues, "*") > -1;
    }

    private static boolean modified(long modifiedHeader, long lastModified) {
        return modifiedHeader + ONE_SECOND_IN_MILLIS <= lastModified;
    }

    private static long sublong(String value, int beginIndex, int endIndex) {
        String substring = value.substring(beginIndex, endIndex);
        return substring.isEmpty() ? -1L : Long.parseLong(substring);
    }

    private static boolean accepts(String acceptHeader, String toAccept) {
        Object[] acceptValues = acceptHeader.split("\\s*(,|;)\\s*");
        Arrays.sort(acceptValues);
        return Arrays.binarySearch(acceptValues, toAccept) > -1 || Arrays.binarySearch(acceptValues, toAccept.replaceAll("/.*$", "/*")) > -1 || Arrays.binarySearch(acceptValues, "*/*") > -1;
    }

    public static String encodeURI(String string) {
        if (string == null) {
            return null;
        }
        return AbstractFileServlet.encodeURL(string).replace("+", "%20").replace("%21", "!").replace("%27", "'").replace("%28", "(").replace("%29", ")").replace("%7E", "~");
    }

    public static String encodeURL(String string) {
        if (string == null) {
            return null;
        }
        try {
            return URLEncoder.encode(string, StandardCharsets.UTF_8.name());
        }
        catch (UnsupportedEncodingException e) {
            throw new UnsupportedOperationException(ERROR_UNSUPPORTED_ENCODING, e);
        }
    }

    @SafeVarargs
    public static <T> T coalesce(T ... objects) {
        for (T object : objects) {
            if (object == null) continue;
            return object;
        }
        return null;
    }

    public static boolean startsWithOneOf(String string, String ... prefixes) {
        for (String prefix : prefixes) {
            if (!string.startsWith(prefix)) continue;
            return true;
        }
        return false;
    }

    public static long stream(InputStream input, OutputStream output) throws IOException {
        try (ReadableByteChannel inputChannel = Channels.newChannel(input);){
            long l;
            block13: {
                WritableByteChannel outputChannel = Channels.newChannel(output);
                try {
                    ByteBuffer buffer = ByteBuffer.allocateDirect(10240);
                    long size = 0L;
                    while (inputChannel.read(buffer) != -1) {
                        buffer.flip();
                        size += (long)outputChannel.write(buffer);
                        buffer.clear();
                    }
                    l = size;
                    if (outputChannel == null) break block13;
                }
                catch (Throwable throwable) {
                    if (outputChannel != null) {
                        try {
                            outputChannel.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                outputChannel.close();
            }
            return l;
        }
    }

    public static long stream(File file, OutputStream output, long start, long length) throws IOException {
        if (start == 0L && length >= file.length()) {
            return AbstractFileServlet.stream(new FileInputStream(file), output);
        }
        try (FileChannel fileChannel = (FileChannel)Files.newByteChannel(file.toPath(), StandardOpenOption.READ);){
            WritableByteChannel outputChannel = Channels.newChannel(output);
            ByteBuffer buffer = ByteBuffer.allocateDirect(10240);
            long size = 0L;
            while (fileChannel.read(buffer, start + size) != -1) {
                buffer.flip();
                if (size + (long)buffer.limit() > length) {
                    buffer.limit((int)(length - size));
                }
                if ((size += (long)outputChannel.write(buffer)) >= length) break;
                buffer.clear();
            }
            long l = size;
            return l;
        }
    }

    public static void setCacheHeaders(HttpServletResponse response, long expires) {
        if (expires > 0L) {
            response.setHeader("Cache-Control", "public,max-age=" + expires + ",must-revalidate");
            long current = System.currentTimeMillis();
            long expiresMillis = TimeUnit.SECONDS.toMillis(expires);
            long expiresValue = current + expiresMillis;
            response.setDateHeader("Expires", expiresValue);
            response.setHeader("Pragma", "");
        } else {
            AbstractFileServlet.setNoCacheHeaders(response);
        }
    }

    public static void setNoCacheHeaders(HttpServletResponse response) {
        response.setHeader("Cache-Control", "no-cache,no-store,must-revalidate");
        response.setDateHeader("Expires", 0L);
        response.setHeader("Pragma", "no-cache");
    }

    public static class Resource {
        private final File file;
        private final long length;
        private final long lastModified;
        private final String eTag;

        public Resource(File file) {
            if (file != null && file.isFile()) {
                this.file = file;
                this.length = file.length();
                this.lastModified = file.lastModified();
                this.eTag = String.format(AbstractFileServlet.ETAG, AbstractFileServlet.encodeURL(file.getName()), this.lastModified);
            } else {
                this.file = null;
                this.length = 0L;
                this.lastModified = 0L;
                this.eTag = null;
            }
        }
    }

    protected class RedirectException
    extends Exception {
        public final String location;

        public RedirectException(String location) {
            this.location = location;
        }
    }

    public static class Range {
        private final long start;
        private final long end;
        private final long length;

        public Range(long start, long end) {
            this.start = start;
            this.end = end;
            this.length = end - start + 1L;
        }
    }
}

