/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.container.security.basic;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import java.util.logging.Logger;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

public class PasswordStorage {
    private static final Logger LOG = Logger.getLogger(PasswordStorage.class.getName());
    public static final String PBKDF2_ALGORITHM = "PBKDF2WithHmacSHA1";
    public static final int SALT_BYTE_SIZE = 24;
    public static final int HASH_BYTE_SIZE = 18;
    public static final int PBKDF2_ITERATIONS = 1000;
    public static final int HASH_SECTIONS = 5;
    public static final int HASH_ALGORITHM_INDEX = 0;
    public static final int ITERATION_INDEX = 1;
    public static final int HASH_SIZE_INDEX = 2;
    public static final int SALT_INDEX = 3;
    public static final int PBKDF2_INDEX = 4;

    public static String createHash(String password) throws CannotPerformOperationException {
        return PasswordStorage.createHash(password.toCharArray());
    }

    public static String createHash(char[] password) throws CannotPerformOperationException {
        SecureRandom random = new SecureRandom();
        byte[] salt = new byte[24];
        random.nextBytes(salt);
        byte[] hash = PasswordStorage.pbkdf2(password, salt, 1000, 18);
        int hashSize = hash.length;
        String parts = "sha1:1000:" + hashSize + ":" + PasswordStorage.toBase64(salt) + ":" + PasswordStorage.toBase64(hash);
        return parts;
    }

    public static boolean verifyPassword(String password, String correctHash) throws CannotPerformOperationException, InvalidHashException {
        return PasswordStorage.verifyPassword(password.toCharArray(), correctHash);
    }

    public static boolean verifyPassword(char[] password, String correctHash) throws CannotPerformOperationException, InvalidHashException {
        String[] params = correctHash.split(":");
        if (params.length != 5) {
            throw new InvalidHashException("Fields are missing from the password hash.");
        }
        if (!params[0].equals("sha1")) {
            throw new CannotPerformOperationException("Unsupported hash type.");
        }
        int iterations = 0;
        try {
            iterations = Integer.parseInt(params[1]);
        }
        catch (NumberFormatException ex) {
            throw new InvalidHashException("Could not parse the iteration count as an integer.", ex);
        }
        if (iterations < 1) {
            throw new InvalidHashException("Invalid number of iterations. Must be >= 1.");
        }
        byte[] salt = null;
        try {
            salt = PasswordStorage.fromBase64(params[3]);
        }
        catch (IllegalArgumentException ex) {
            throw new InvalidHashException("Base64 decoding of salt failed.", ex);
        }
        byte[] hash = null;
        try {
            hash = PasswordStorage.fromBase64(params[4]);
        }
        catch (IllegalArgumentException ex) {
            throw new InvalidHashException("Base64 decoding of pbkdf2 output failed.", ex);
        }
        int storedHashSize = 0;
        try {
            storedHashSize = Integer.parseInt(params[2]);
        }
        catch (NumberFormatException ex) {
            throw new InvalidHashException("Could not parse the hash size as an integer.", ex);
        }
        if (storedHashSize != hash.length) {
            throw new InvalidHashException("Hash length doesn't match stored hash length.");
        }
        byte[] testHash = PasswordStorage.pbkdf2(password, salt, iterations, hash.length);
        return PasswordStorage.slowEquals(hash, testHash);
    }

    private static boolean slowEquals(byte[] a, byte[] b) {
        int diff = a.length ^ b.length;
        for (int i = 0; i < a.length && i < b.length; ++i) {
            diff |= a[i] ^ b[i];
        }
        return diff == 0;
    }

    private static byte[] pbkdf2(char[] password, byte[] salt, int iterations, int bytes) {
        try {
            LOG.fine("Cache miss, hashing password...");
            PBEKeySpec spec = new PBEKeySpec(password, salt, iterations, bytes * 8);
            SecretKeyFactory skf = SecretKeyFactory.getInstance(PBKDF2_ALGORITHM);
            return skf.generateSecret(spec).getEncoded();
        }
        catch (NoSuchAlgorithmException ex) {
            throw new CannotPerformOperationException("Hash algorithm not supported.", ex);
        }
        catch (InvalidKeySpecException ex) {
            throw new CannotPerformOperationException("Invalid key spec.", ex);
        }
    }

    private static byte[] fromBase64(String hex) {
        return Base64.getDecoder().decode(hex);
    }

    private static String toBase64(byte[] array) {
        return Base64.getEncoder().encodeToString(array);
    }

    public static class InvalidHashException
    extends RuntimeException {
        public InvalidHashException(String message) {
            super(message);
        }

        public InvalidHashException(String message, Throwable source) {
            super(message, source);
        }
    }

    public static class CannotPerformOperationException
    extends RuntimeException {
        public CannotPerformOperationException(String message) {
            super(message);
        }

        public CannotPerformOperationException(String message, Throwable source) {
            super(message, source);
        }
    }
}

