/*
 * Decompiled with CFR 0.152.
 */
package org.openremote.agent.protocol.velbus.device;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.openremote.agent.protocol.velbus.AbstractVelbusProtocol;
import org.openremote.agent.protocol.velbus.VelbusNetwork;
import org.openremote.agent.protocol.velbus.VelbusPacket;
import org.openremote.agent.protocol.velbus.device.FeatureProcessor;
import org.openremote.agent.protocol.velbus.device.VelbusDeviceType;
import org.openremote.model.util.TextUtil;

public class VelbusDevice {
    public static final int MAX_INITIALISATION_ATTEMPTS = 5;
    public static int INITIALISATION_TIMEOUT_MILLISECONDS = 10000;
    protected int baseAddress;
    protected int[] subAddresses = new int[4];
    protected final Map<String, Object> devicePropertyCache = new HashMap<String, Object>();
    protected final Map<String, List<Consumer<Object>>> propertyValueConsumers = new ConcurrentHashMap<String, List<Consumer<Object>>>();
    protected VelbusNetwork velbusNetwork;
    protected FeatureProcessor[] featureProcessors;
    protected boolean initialised;
    protected boolean initialisationFailed;
    protected int initialisationAttempts;
    protected VelbusDeviceType deviceType;
    protected Future<?> initialisationTask;

    public VelbusDevice(int baseAddress, VelbusNetwork velbusNetwork) {
        this.baseAddress = baseAddress;
        this.velbusNetwork = velbusNetwork;
    }

    public int getBaseAddress() {
        return this.baseAddress;
    }

    public int[] getSubAddresses() {
        return this.subAddresses;
    }

    public VelbusDeviceType getDeviceType() {
        return this.deviceType;
    }

    public int getAddress(int index) {
        return (index = Math.max(0, Math.min(5, index))) == 0 ? this.getBaseAddress() : this.subAddresses[index - 1];
    }

    public int getAddressIndex(int address) {
        if (this.baseAddress == address) {
            return 0;
        }
        for (int i = 0; i < this.subAddresses.length; ++i) {
            if (this.subAddresses[i] != address) continue;
            return i + 1;
        }
        return -1;
    }

    private void setDeviceType(VelbusDeviceType deviceType) {
        this.deviceType = deviceType;
        this.featureProcessors = deviceType.getFeatureProcessors();
    }

    private void cancelInitialisationTask(boolean interrupt) {
        if (this.initialisationTask != null) {
            this.initialisationTask.cancel(interrupt);
            this.initialisationTask = null;
        }
    }

    public void reset() {
        this.cancelInitialisationTask(true);
        this.devicePropertyCache.clear();
        this.initialised = false;
        this.initialisationAttempts = 0;
        this.deviceType = null;
        Arrays.fill(this.subAddresses, 0);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void initialise() {
        VelbusDevice velbusDevice = this;
        synchronized (velbusDevice) {
            if (this.isInitialised()) {
                return;
            }
            if (this.initialisationTask != null) {
                AbstractVelbusProtocol.LOG.finest("Initialisation already in progress");
                return;
            }
            if (this.initialisationFailed) {
                this.initialisationAttempts = 0;
                this.initialisationFailed = false;
                AbstractVelbusProtocol.LOG.finest("Re-attempting device initialisation");
            }
            AbstractVelbusProtocol.LOG.info("Initialisation starting: " + this.getBaseAddress());
            this.initialisationTask = this.velbusNetwork.getExecutorService().scheduleWithFixedDelay(this::doInitialisation, 0L, INITIALISATION_TIMEOUT_MILLISECONDS, TimeUnit.MILLISECONDS);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doInitialisation() {
        VelbusDevice velbusDevice = this;
        synchronized (velbusDevice) {
            if (this.initialisationAttempts >= 5) {
                AbstractVelbusProtocol.LOG.fine("Initialisation failed - Device has reached maximum initialisation attempts: " + this.getBaseAddress());
                this.initialisationFailed = true;
                this.cancelInitialisationTask(false);
                return;
            }
            ++this.initialisationAttempts;
            this.velbusNetwork.sendPackets(VelbusDevice.createModuleTypePacket(this.baseAddress));
        }
    }

    public boolean isInitialised() {
        return this.initialised;
    }

    public boolean isInitialisedAndValid() {
        return this.isInitialised() && this.deviceType != VelbusDeviceType.UNKNOWN;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onInitialised() {
        VelbusDevice velbusDevice = this;
        synchronized (velbusDevice) {
            if (this.isInitialised()) {
                return;
            }
            this.cancelInitialisationTask(true);
            AbstractVelbusProtocol.LOG.info("Device initialised: " + this.getBaseAddress());
            this.initialised = true;
        }
        if (this.isInitialisedAndValid() && this.featureProcessors != null) {
            AbstractVelbusProtocol.LOG.finer("Sending module status request packets");
            this.velbusNetwork.sendPackets((VelbusPacket[])Arrays.stream(this.featureProcessors).flatMap(processor -> processor.getStatusRequestPackets(this).stream()).distinct().toArray(VelbusPacket[]::new));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addPropertyValueConsumer(String property, Consumer<Object> propertyValueConsumer) {
        List consumers;
        if (property.isEmpty()) {
            return;
        }
        List list = consumers = this.propertyValueConsumers.computeIfAbsent(property, p -> new ArrayList());
        synchronized (list) {
            consumers.add(propertyValueConsumer);
        }
        propertyValueConsumer.accept(this.getPropertyValue(property));
    }

    public void removePropertyValueConsumer(String property, Consumer<Object> propertyValueConsumer) {
        if (property.isEmpty()) {
            return;
        }
        this.propertyValueConsumers.computeIfPresent(property, (prop, consumers) -> {
            List list = consumers;
            synchronized (list) {
                consumers.remove(propertyValueConsumer);
            }
            return consumers;
        });
    }

    public void removeAllPropertyValueConsumers() {
        this.propertyValueConsumers.forEach((prop, consumers) -> consumers.clear());
        this.propertyValueConsumers.clear();
    }

    public void writeProperty(String property, Object value) {
        if (!this.isInitialisedAndValid()) {
            AbstractVelbusProtocol.LOG.finer("Ignoring property write as device is not initialised and/or it is invalid");
            return;
        }
        if (TextUtil.isNullOrEmpty((String)property)) {
            return;
        }
        if (this.featureProcessors != null) {
            for (FeatureProcessor processor : this.featureProcessors) {
                List<VelbusPacket> packets = processor.getPropertyWritePackets(this, property, value);
                if (packets == null) continue;
                this.velbusNetwork.sendPackets(packets.toArray(new VelbusPacket[0]));
                break;
            }
        }
    }

    public void processReceivedPacket(VelbusPacket velbusPacket) {
        VelbusPacket.InboundCommand packetCommand = VelbusPacket.InboundCommand.fromCode(velbusPacket.getCommand());
        switch (packetCommand) {
            case UNKNOWN: {
                AbstractVelbusProtocol.LOG.info("Packet received and ignored for device '" + this.baseAddress + "': " + Integer.toHexString(velbusPacket.getCommand()) + " (" + packetCommand + ")");
                break;
            }
            case MODULE_TYPE: {
                int typeCode = velbusPacket.getTypeCode();
                this.setDeviceType(VelbusDeviceType.fromCode(typeCode));
                if (!this.deviceType.hasSubAddresses()) {
                    this.onInitialised();
                }
                AbstractVelbusProtocol.LOG.finest("Packet received and handled by device '" + this.baseAddress + "': " + packetCommand);
                break;
            }
            case MODULE_SUBADDRESSES: {
                if (velbusPacket.getDataSize() != 8) {
                    AbstractVelbusProtocol.LOG.warning("MODULE_SUBADDRESSES packet doesn't contain exactly 8 data bytes");
                    break;
                }
                for (int i = 4; i < velbusPacket.getDataSize(); ++i) {
                    int subAddress;
                    this.subAddresses[i - 4] = subAddress = velbusPacket.getInt(i);
                    if (subAddress == 255) continue;
                    this.velbusNetwork.registerSubAddress(this, subAddress);
                }
                AbstractVelbusProtocol.LOG.finest("Packet received and handled by device '" + this.baseAddress + "': " + packetCommand);
                this.onInitialised();
                break;
            }
            default: {
                if (!this.isInitialisedAndValid()) {
                    return;
                }
                boolean handled = false;
                if (this.featureProcessors != null) {
                    handled = Arrays.stream(this.featureProcessors).anyMatch(processor -> processor.processReceivedPacket(this, velbusPacket));
                }
                boolean bl = handled = handled || velbusPacket.isHandled();
                if (!handled) {
                    AbstractVelbusProtocol.LOG.fine("Packet received was not handled by device '" + this.baseAddress + "': " + packetCommand);
                    break;
                }
                AbstractVelbusProtocol.LOG.finest("Packet received and handled by device '" + this.baseAddress + "': " + packetCommand);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void setProperty(String property, Object value) {
        property = property.toUpperCase();
        Map<String, Object> map = this.devicePropertyCache;
        synchronized (map) {
            this.devicePropertyCache.put(property, value);
        }
        this.propertyValueConsumers.computeIfPresent(property, (prop, consumers) -> {
            consumers.forEach(consumer -> consumer.accept(value));
            return consumers;
        });
    }

    protected Object getPropertyValue(String propertyName) {
        return this.devicePropertyCache.get(propertyName);
    }

    protected boolean hasPropertyValue(String propertyName) {
        return this.devicePropertyCache.containsKey(propertyName);
    }

    public static VelbusPacket[] createTimeInjectionPackets() {
        Calendar c = Calendar.getInstance();
        boolean dst = c.get(16) > 0;
        int dow = (c.get(7) + 5) % 7;
        int dom = c.get(5);
        int month = c.get(2) + 1;
        int hours = c.get(11);
        int mins = c.get(12);
        int year = c.get(1);
        VelbusPacket[] packets = new VelbusPacket[]{new VelbusPacket(0, VelbusPacket.OutboundCommand.REALTIME_CLOCK_SET.getCode(), (byte)dow, (byte)hours, (byte)mins), new VelbusPacket(0, VelbusPacket.OutboundCommand.REALTIME_DATE_SET.getCode(), (byte)dom, (byte)month, (byte)(year >>> 8), (byte)year), new VelbusPacket(0, VelbusPacket.OutboundCommand.DAYLIGHT_SAVING_SET.getCode(), (byte)(dst ? 1 : 0))};
        return packets;
    }

    public static VelbusPacket createModuleTypePacket(int baseAddress) {
        return new VelbusPacket(baseAddress, VelbusPacket.PacketPriority.LOW, 0, true);
    }
}

