/*
 * Copyright 2021, OpenRemote Inc.
 *
 * See the CONTRIBUTORS.txt file in the distribution for a
 * full listing of individual contributors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.openremote.agent.protocol.bluetooth.mesh.transport;

import org.openremote.agent.protocol.bluetooth.mesh.opcodes.ConfigMessageOpCodes;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.logging.Logger;

import static org.openremote.agent.protocol.bluetooth.mesh.utils.MeshAddress.addressIntToBytes;
import static org.openremote.agent.protocol.bluetooth.mesh.utils.MeshAddress.isValidUnassignedAddress;
import static org.openremote.agent.protocol.bluetooth.mesh.utils.MeshAddress.isValidUnicastAddress;
import static org.openremote.agent.protocol.bluetooth.mesh.utils.MeshAddress.isValidVirtualAddress;

/**
 * Creates the ConfigModelSubscriptionDelete message
 */
@SuppressWarnings({"unused"})
public final class ConfigModelSubscriptionDelete extends ConfigMessage {

    public static final Logger LOG = Logger.getLogger(ConfigModelSubscriptionDelete.class.getName());
    private static final int OP_CODE = ConfigMessageOpCodes.CONFIG_MODEL_SUBSCRIPTION_DELETE;

    private static final int SIG_MODEL_APP_KEY_BIND_PARAMS_LENGTH = 6;
    private static final int VENDOR_MODEL_APP_KEY_BIND_PARAMS_LENGTH = 8;

    private final int mElementAddress;
    private final int mSubscriptionAddress;
    private final int mModelIdentifier;

    /**
     * Constructs ConfigModelSubscriptionDelete message.
     *
     * @param elementAddress      Address of the element to which the model belongs to.
     * @param subscriptionAddress Address to unsubscribe from or deleted.
     * @param modelIdentifier     identifier of the model, 16-bit for Sig model and 32-bit model id for vendor models.
     * @throws IllegalArgumentException if any illegal arguments are passed
     */
    public ConfigModelSubscriptionDelete(final int elementAddress,
                                         final int subscriptionAddress,
                                         final int modelIdentifier) throws IllegalArgumentException {

        if (!isValidUnicastAddress(elementAddress))
            throw new IllegalArgumentException("Invalid unicast address, unicast address must be a 16-bit value, and must range from 0x0001 to 0x7FFF");
        this.mElementAddress = elementAddress;
        if (isValidUnassignedAddress(subscriptionAddress) || isValidUnicastAddress(subscriptionAddress) || isValidVirtualAddress(subscriptionAddress) || subscriptionAddress == 0xFFFF)
            throw new IllegalArgumentException("The value of the Address field shall not be an unassigned address, unicast address, " +
                "all-nodes address or virtual address.");
        this.mSubscriptionAddress = subscriptionAddress;
        this.mModelIdentifier = modelIdentifier;
        assembleMessageParameters();
    }

    @Override
    public int getOpCode() {
        return OP_CODE;
    }

    @Override
    void assembleMessageParameters() {

        final ByteBuffer paramsBuffer;
        //We check if the model identifier value is within the range of a 16-bit value here. If it is then it is a sigmodel
        final byte[] elementaddress = addressIntToBytes(mElementAddress);
        final byte[] subscriptionAddress = addressIntToBytes(mSubscriptionAddress);
        if (mModelIdentifier >= Short.MIN_VALUE && mModelIdentifier <= Short.MAX_VALUE) {
            paramsBuffer = ByteBuffer.allocate(SIG_MODEL_APP_KEY_BIND_PARAMS_LENGTH).order(ByteOrder.LITTLE_ENDIAN);
            paramsBuffer.put(elementaddress[1]);
            paramsBuffer.put(elementaddress[0]);
            paramsBuffer.put(subscriptionAddress[1]);
            paramsBuffer.put(subscriptionAddress[0]);
            paramsBuffer.putShort((short) mModelIdentifier);
            mParameters = paramsBuffer.array();
        } else {
            paramsBuffer = ByteBuffer.allocate(VENDOR_MODEL_APP_KEY_BIND_PARAMS_LENGTH).order(ByteOrder.LITTLE_ENDIAN);
            paramsBuffer.put(elementaddress[1]);
            paramsBuffer.put(elementaddress[0]);
            paramsBuffer.put(subscriptionAddress[1]);
            paramsBuffer.put(subscriptionAddress[0]);
            final byte[] modelIdentifier = new byte[]{(byte) ((mModelIdentifier >> 24) & 0xFF),
                (byte) ((mModelIdentifier >> 16) & 0xFF), (byte) ((mModelIdentifier >> 8) & 0xFF), (byte) (mModelIdentifier & 0xFF)};
            paramsBuffer.put(modelIdentifier[1]);
            paramsBuffer.put(modelIdentifier[0]);
            paramsBuffer.put(modelIdentifier[3]);
            paramsBuffer.put(modelIdentifier[2]);
            mParameters = paramsBuffer.array();
        }
    }
}
