package io.openlineage.server;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import io.openlineage.server.OpenLineage.RunEvent.EventType;
import java.lang.Object;
import java.lang.String;
import java.net.URI;
import java.time.ZonedDateTime;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public final class OpenLineage {
  public interface Builder<T> {
    /**
     * @return the constructed type
     */
    T build();
  }

  @JsonPropertyOrder
  public static final class RunFacets {
    @JsonAnySetter
    private final Map<String, RunFacet> additionalProperties;

    @JsonCreator
    public RunFacets() {
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, RunFacet> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "eventType",
      "eventTime",
      "run",
      "job",
      "inputs",
      "outputs",
      "producer",
      "schemaURL"
  })
  public static final class RunEvent {
    private final RunEvent.EventType eventType;

    private final ZonedDateTime eventTime;

    private final Run run;

    private final Job job;

    private final List<InputDataset> inputs;

    private final List<OutputDataset> outputs;

    private final URI producer;

    private final URI schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param eventType the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
     * @param eventTime the time the event occurred at
     * @param run the run
     * @param job the job
     * @param inputs The set of **input** datasets.
     * @param outputs The set of **output** datasets.
     * @param producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this RunEvent
     */
    @JsonCreator
    public RunEvent(@JsonProperty("eventType") RunEvent.EventType eventType,
        @JsonProperty("eventTime") ZonedDateTime eventTime, @JsonProperty("run") Run run,
        @JsonProperty("job") Job job, @JsonProperty("inputs") List<InputDataset> inputs,
        @JsonProperty("outputs") List<OutputDataset> outputs,
        @JsonProperty("producer") URI producer, @JsonProperty("schemaURL") URI schemaURL) {
      this.eventType = eventType;
      this.eventTime = eventTime;
      this.run = run;
      this.job = job;
      this.inputs = inputs;
      this.outputs = outputs;
      this.producer = producer;
      this.schemaURL = schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
     */
    public RunEvent.EventType getEventType() {
      return eventType;
    }

    /**
     * @return the time the event occurred at
     */
    public ZonedDateTime getEventTime() {
      return eventTime;
    }

    public Run getRun() {
      return run;
    }

    public Job getJob() {
      return job;
    }

    /**
     * @return The set of **input** datasets.
     */
    public List<InputDataset> getInputs() {
      return inputs;
    }

    /**
     * @return The set of **output** datasets.
     */
    public List<OutputDataset> getOutputs() {
      return outputs;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI getProducer() {
      return producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this RunEvent
     */
    public URI getSchemaURL() {
      return schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }

    public enum EventType {
      START,

      COMPLETE,

      ABORT,

      FAIL,

      OTHER
    }
  }

  @JsonPropertyOrder
  public static final class JobFacets {
    @JsonAnySetter
    private final Map<String, JobFacet> additionalProperties;

    @JsonCreator
    public JobFacets() {
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, JobFacet> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder
  public static final class InputDatasetInputFacets {
    @JsonAnySetter
    private final Map<String, InputDatasetFacet> additionalProperties;

    @JsonCreator
    public InputDatasetInputFacets() {
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, InputDatasetFacet> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "_producer",
      "_schemaURL"
  })
  public static final class DatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param _schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    @JsonCreator
    public DatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("_schemaURL") URI _schemaURL) {
      this._producer = _producer;
      this._schemaURL = _schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "_producer",
      "_schemaURL"
  })
  public static final class OutputDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param _schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    @JsonCreator
    public OutputDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("_schemaURL") URI _schemaURL) {
      this._producer = _producer;
      this._schemaURL = _schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder
  public static final class DatasetFacets {
    @JsonAnySetter
    private final Map<String, DatasetFacet> additionalProperties;

    @JsonCreator
    public DatasetFacets() {
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, DatasetFacet> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "runId",
      "facets"
  })
  public static final class Run {
    private final UUID runId;

    private final RunFacets facets;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param runId The globally unique ID of the run associated with the job.
     * @param facets The run facets.
     */
    @JsonCreator
    public Run(@JsonProperty("runId") UUID runId, @JsonProperty("facets") RunFacets facets) {
      this.runId = runId;
      this.facets = facets;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return The globally unique ID of the run associated with the job.
     */
    public UUID getRunId() {
      return runId;
    }

    /**
     * @return The run facets.
     */
    public RunFacets getFacets() {
      return facets;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  public interface Dataset {
    /**
     * @return The namespace containing that dataset
     */
    String getNamespace();

    /**
     * @return The unique name for that dataset within that namespace
     */
    String getName();

    /**
     * @return The facets for this dataset
     */
    DatasetFacets getFacets();

    /**
     * @return additional properties
     */
    Map<String, Object> getAdditionalProperties();
  }

  @JsonPropertyOrder
  public static final class OutputDatasetOutputFacets {
    @JsonAnySetter
    private final Map<String, OutputDatasetFacet> additionalProperties;

    @JsonCreator
    public OutputDatasetOutputFacets() {
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, OutputDatasetFacet> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "_producer",
      "_schemaURL"
  })
  public static final class RunFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param _schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    @JsonCreator
    public RunFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("_schemaURL") URI _schemaURL) {
      this._producer = _producer;
      this._schemaURL = _schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "namespace",
      "name",
      "facets",
      "outputFacets"
  })
  public static final class OutputDataset implements Dataset {
    private final String namespace;

    private final String name;

    private final DatasetFacets facets;

    private final OutputDatasetOutputFacets outputFacets;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param namespace The namespace containing that dataset
     * @param name The unique name for that dataset within that namespace
     * @param facets The facets for this dataset
     * @param outputFacets The output facets for this dataset
     */
    @JsonCreator
    public OutputDataset(@JsonProperty("namespace") String namespace,
        @JsonProperty("name") String name, @JsonProperty("facets") DatasetFacets facets,
        @JsonProperty("outputFacets") OutputDatasetOutputFacets outputFacets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
      this.outputFacets = outputFacets;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return The namespace containing that dataset
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that dataset within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The facets for this dataset
     */
    public DatasetFacets getFacets() {
      return facets;
    }

    /**
     * @return The output facets for this dataset
     */
    public OutputDatasetOutputFacets getOutputFacets() {
      return outputFacets;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  public interface BaseFacet {
    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    URI get_producer();

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    URI get_schemaURL();

    /**
     * @return additional properties
     */
    Map<String, Object> getAdditionalProperties();
  }

  @JsonPropertyOrder({
      "namespace",
      "name",
      "facets",
      "inputFacets"
  })
  public static final class InputDataset implements Dataset {
    private final String namespace;

    private final String name;

    private final DatasetFacets facets;

    private final InputDatasetInputFacets inputFacets;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param namespace The namespace containing that dataset
     * @param name The unique name for that dataset within that namespace
     * @param facets The facets for this dataset
     * @param inputFacets The input facets for this dataset.
     */
    @JsonCreator
    public InputDataset(@JsonProperty("namespace") String namespace,
        @JsonProperty("name") String name, @JsonProperty("facets") DatasetFacets facets,
        @JsonProperty("inputFacets") InputDatasetInputFacets inputFacets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
      this.inputFacets = inputFacets;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return The namespace containing that dataset
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that dataset within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The facets for this dataset
     */
    public DatasetFacets getFacets() {
      return facets;
    }

    /**
     * @return The input facets for this dataset.
     */
    public InputDatasetInputFacets getInputFacets() {
      return inputFacets;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "namespace",
      "name",
      "facets"
  })
  public static final class Job {
    private final String namespace;

    private final String name;

    private final JobFacets facets;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param namespace The namespace containing that job
     * @param name The unique name for that job within that namespace
     * @param facets The job facets.
     */
    @JsonCreator
    public Job(@JsonProperty("namespace") String namespace, @JsonProperty("name") String name,
        @JsonProperty("facets") JobFacets facets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return The namespace containing that job
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that job within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The job facets.
     */
    public JobFacets getFacets() {
      return facets;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "_producer",
      "_schemaURL"
  })
  public static final class InputDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param _schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    @JsonCreator
    public InputDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("_schemaURL") URI _schemaURL) {
      this._producer = _producer;
      this._schemaURL = _schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }

  @JsonPropertyOrder({
      "_producer",
      "_schemaURL"
  })
  public static final class JobFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param _schemaURL The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    @JsonCreator
    public JobFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("_schemaURL") URI _schemaURL) {
      this._producer = _producer;
      this._schemaURL = _schemaURL;
      this.additionalProperties = new LinkedHashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }

    /**
     * Get object with additional properties
     */
    void withAdditionalProperties() {
    }
  }
}
