package io.openlineage.client;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.lang.Double;
import java.lang.Long;
import java.lang.Object;
import java.lang.String;
import java.net.URI;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public final class OpenLineage {
  private final URI producer;

  public OpenLineage(URI producer) {
    this.producer = producer;
  }

  /**
   * @param runId The globally unique ID of the run associated with the job.
   * @return ParentRunFacetRun
   */
  public ParentRunFacetRun newParentRunFacetRun(UUID runId) {
    return new ParentRunFacetRun(runId);
  }

  public ParentRunFacetRunBuilder newParentRunFacetRunBuilder() {
    return new ParentRunFacetRunBuilder();
  }

  /**
   * @param name the name
   * @param uri the uri
   * @return DatasourceDatasetFacet
   */
  public DatasourceDatasetFacet newDatasourceDatasetFacet(String name, URI uri) {
    return new DatasourceDatasetFacet(this.producer, name, uri);
  }

  public DatasourceDatasetFacetBuilder newDatasourceDatasetFacetBuilder() {
    return new DatasourceDatasetFacetBuilder();
  }

  /**
   * @param description The description of the job.
   * @return DocumentationJobFacet
   */
  public DocumentationJobFacet newDocumentationJobFacet(String description) {
    return new DocumentationJobFacet(this.producer, description);
  }

  public DocumentationJobFacetBuilder newDocumentationJobFacetBuilder() {
    return new DocumentationJobFacetBuilder();
  }

  /**
   * @param rowCount The number of rows written to the dataset
   * @param size The size in bytes written to the dataset
   * @return OutputStatisticsOutputDatasetFacet
   */
  public OutputStatisticsOutputDatasetFacet newOutputStatisticsOutputDatasetFacet(Long rowCount,
      Long size) {
    return new OutputStatisticsOutputDatasetFacet(this.producer, rowCount, size);
  }

  public OutputStatisticsOutputDatasetFacetBuilder newOutputStatisticsOutputDatasetFacetBuilder() {
    return new OutputStatisticsOutputDatasetFacetBuilder();
  }

  /**
   * @param rowCount The number of rows evaluated
   * @param bytes The size in bytes
   * @param columnMetrics The property key is the column name
   * @return DataQualityMetricsInputDatasetFacet
   */
  public DataQualityMetricsInputDatasetFacet newDataQualityMetricsInputDatasetFacet(Long rowCount,
      Long bytes, DataQualityMetricsInputDatasetFacetColumnMetrics columnMetrics) {
    return new DataQualityMetricsInputDatasetFacet(this.producer, rowCount, bytes, columnMetrics);
  }

  public DataQualityMetricsInputDatasetFacetBuilder newDataQualityMetricsInputDatasetFacetBuilder(
      ) {
    return new DataQualityMetricsInputDatasetFacetBuilder();
  }

  /**
   * @param nominalTime the nominalTime
   * @param parent the parent
   * @return RunFacets
   */
  public RunFacets newRunFacets(NominalTimeRunFacet nominalTime, ParentRunFacet parent) {
    return new RunFacets(nominalTime, parent);
  }

  public RunFacetsBuilder newRunFacetsBuilder() {
    return new RunFacetsBuilder();
  }

  /**
   * @param documentation the documentation
   * @param sourceCodeLocation the sourceCodeLocation
   * @param sql the sql
   * @return JobFacets
   */
  public JobFacets newJobFacets(DocumentationJobFacet documentation,
      SourceCodeLocationJobFacet sourceCodeLocation, SQLJobFacet sql) {
    return new JobFacets(documentation, sourceCodeLocation, sql);
  }

  public JobFacetsBuilder newJobFacetsBuilder() {
    return new JobFacetsBuilder();
  }

  /**
   * @param namespace The namespace containing that job
   * @param name The unique name for that job within that namespace
   * @param facets The job facets.
   * @return Job
   */
  public Job newJob(String namespace, String name, JobFacets facets) {
    return new Job(namespace, name, facets);
  }

  public JobBuilder newJobBuilder() {
    return new JobBuilder();
  }

  /**
   * @param namespace The namespace containing that job
   * @param name The unique name for that job within that namespace
   * @return ParentRunFacetJob
   */
  public ParentRunFacetJob newParentRunFacetJob(String namespace, String name) {
    return new ParentRunFacetJob(namespace, name);
  }

  public ParentRunFacetJobBuilder newParentRunFacetJobBuilder() {
    return new ParentRunFacetJobBuilder();
  }

  /**
   * @param description The description of the dataset.
   * @return DocumentationDatasetFacet
   */
  public DocumentationDatasetFacet newDocumentationDatasetFacet(String description) {
    return new DocumentationDatasetFacet(this.producer, description);
  }

  public DocumentationDatasetFacetBuilder newDocumentationDatasetFacetBuilder() {
    return new DocumentationDatasetFacetBuilder();
  }

  /**
   * @param type the source control system
   * @param url the full http URL to locate the file
   * @param repoUrl the URL to the repository
   * @param path the path in the repo containing the source files
   * @param version the current version deployed (not a branch name, the actual unique version)
   * @param tag optional tag name
   * @param branch optional branch name
   * @return SourceCodeLocationJobFacet
   */
  public SourceCodeLocationJobFacet newSourceCodeLocationJobFacet(String type, URI url,
      String repoUrl, String path, String version, String tag, String branch) {
    return new SourceCodeLocationJobFacet(this.producer, type, url, repoUrl, path, version, tag, branch);
  }

  public SourceCodeLocationJobFacetBuilder newSourceCodeLocationJobFacetBuilder() {
    return new SourceCodeLocationJobFacetBuilder();
  }

  /**
   * @param run the run
   * @param job the job
   * @return ParentRunFacet
   */
  public ParentRunFacet newParentRunFacet(ParentRunFacetRun run, ParentRunFacetJob job) {
    return new ParentRunFacet(this.producer, run, job);
  }

  public ParentRunFacetBuilder newParentRunFacetBuilder() {
    return new ParentRunFacetBuilder();
  }

  /**
   * @param outputStatistics the outputStatistics
   * @return OutputDatasetOutputFacets
   */
  public OutputDatasetOutputFacets newOutputDatasetOutputFacets(
      OutputStatisticsOutputDatasetFacet outputStatistics) {
    return new OutputDatasetOutputFacets(outputStatistics);
  }

  public OutputDatasetOutputFacetsBuilder newOutputDatasetOutputFacetsBuilder() {
    return new OutputDatasetOutputFacetsBuilder();
  }

  /**
   * @param documentation the documentation
   * @param schema the schema
   * @param dataSource the dataSource
   * @return DatasetFacets
   */
  public DatasetFacets newDatasetFacets(DocumentationDatasetFacet documentation,
      SchemaDatasetFacet schema, DatasourceDatasetFacet dataSource) {
    return new DatasetFacets(documentation, schema, dataSource);
  }

  public DatasetFacetsBuilder newDatasetFacetsBuilder() {
    return new DatasetFacetsBuilder();
  }

  /**
   * @param nominalStartTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal start time (included) of the run. AKA the schedule time
   * @param nominalEndTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal end time (excluded) of the run. (Should be the nominal start time of the next run)
   * @return NominalTimeRunFacet
   */
  public NominalTimeRunFacet newNominalTimeRunFacet(ZonedDateTime nominalStartTime,
      ZonedDateTime nominalEndTime) {
    return new NominalTimeRunFacet(this.producer, nominalStartTime, nominalEndTime);
  }

  public NominalTimeRunFacetBuilder newNominalTimeRunFacetBuilder() {
    return new NominalTimeRunFacetBuilder();
  }

  /**
   * @param fields The fields of the table.
   * @return SchemaDatasetFacet
   */
  public SchemaDatasetFacet newSchemaDatasetFacet(List<SchemaDatasetFacetFields> fields) {
    return new SchemaDatasetFacet(this.producer, fields);
  }

  public SchemaDatasetFacetBuilder newSchemaDatasetFacetBuilder() {
    return new SchemaDatasetFacetBuilder();
  }

  /**
   * @param query the query
   * @return SQLJobFacet
   */
  public SQLJobFacet newSQLJobFacet(String query) {
    return new SQLJobFacet(this.producer, query);
  }

  public SQLJobFacetBuilder newSQLJobFacetBuilder() {
    return new SQLJobFacetBuilder();
  }

  /**
   * @param runId The globally unique ID of the run associated with the job.
   * @param facets The run facets.
   * @return Run
   */
  public Run newRun(UUID runId, RunFacets facets) {
    return new Run(runId, facets);
  }

  public RunBuilder newRunBuilder() {
    return new RunBuilder();
  }

  /**
   * @return CustomFacet
   */
  public CustomFacet newCustomFacet() {
    return new CustomFacet(this.producer);
  }

  public CustomFacetBuilder newCustomFacetBuilder() {
    return new CustomFacetBuilder();
  }

  /**
   * @return DataQualityMetricsInputDatasetFacetColumnMetrics
   */
  public DataQualityMetricsInputDatasetFacetColumnMetrics newDataQualityMetricsInputDatasetFacetColumnMetrics(
      ) {
    return new DataQualityMetricsInputDatasetFacetColumnMetrics();
  }

  public DataQualityMetricsInputDatasetFacetColumnMetricsBuilder newDataQualityMetricsInputDatasetFacetColumnMetricsBuilder(
      ) {
    return new DataQualityMetricsInputDatasetFacetColumnMetricsBuilder();
  }

  /**
   * @param namespace The namespace containing that dataset
   * @param name The unique name for that dataset within that namespace
   * @param facets The facets for this dataset
   * @param inputFacets The input facets for this dataset.
   * @return InputDataset
   */
  public InputDataset newInputDataset(String namespace, String name, DatasetFacets facets,
      InputDatasetInputFacets inputFacets) {
    return new InputDataset(namespace, name, facets, inputFacets);
  }

  public InputDatasetBuilder newInputDatasetBuilder() {
    return new InputDatasetBuilder();
  }

  /**
   * @param namespace The namespace containing that dataset
   * @param name The unique name for that dataset within that namespace
   * @param facets The facets for this dataset
   * @param outputFacets The output facets for this dataset
   * @return OutputDataset
   */
  public OutputDataset newOutputDataset(String namespace, String name, DatasetFacets facets,
      OutputDatasetOutputFacets outputFacets) {
    return new OutputDataset(namespace, name, facets, outputFacets);
  }

  public OutputDatasetBuilder newOutputDatasetBuilder() {
    return new OutputDatasetBuilder();
  }

  /**
   * @return DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles
   */
  public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles newDataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles(
      ) {
    return new DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles();
  }

  public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantilesBuilder newDataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantilesBuilder(
      ) {
    return new DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantilesBuilder();
  }

  /**
   * @param nullCount The number of null values in this column for the rows evaluated
   * @param distinctCount The number of distinct values in this column for the rows evaluated
   * @param sum The total sum of values in this column for the rows evaluated
   * @param count The number of values in this column
   * @param min the min
   * @param max the max
   * @param quantiles The property key is the quantile. Examples: 0.1 0.25 0.5 0.75 1
   * @return DataQualityMetricsInputDatasetFacetColumnMetricsAdditional
   */
  public DataQualityMetricsInputDatasetFacetColumnMetricsAdditional newDataQualityMetricsInputDatasetFacetColumnMetricsAdditional(
      Long nullCount, Long distinctCount, Double sum, Double count, Double min, Double max,
      DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles quantiles) {
    return new DataQualityMetricsInputDatasetFacetColumnMetricsAdditional(nullCount, distinctCount, sum, count, min, max, quantiles);
  }

  public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder newDataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder(
      ) {
    return new DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder();
  }

  /**
   * @param eventType the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
   * @param eventTime the time the event occured at
   * @param run the run
   * @param job the job
   * @param inputs The set of **input** datasets.
   * @param outputs The set of **output** datasets.
   * @return RunEvent
   */
  public RunEvent newRunEvent(String eventType, ZonedDateTime eventTime, Run run, Job job,
      List<InputDataset> inputs, List<OutputDataset> outputs) {
    return new RunEvent(eventType, eventTime, run, job, inputs, outputs, this.producer);
  }

  public RunEventBuilder newRunEventBuilder() {
    return new RunEventBuilder();
  }

  /**
   * @param name The name of the field.
   * @param type The type of the field.
   * @param description The description of the field.
   * @return SchemaDatasetFacetFields
   */
  public SchemaDatasetFacetFields newSchemaDatasetFacetFields(String name, String type,
      String description) {
    return new SchemaDatasetFacetFields(name, type, description);
  }

  public SchemaDatasetFacetFieldsBuilder newSchemaDatasetFacetFieldsBuilder() {
    return new SchemaDatasetFacetFieldsBuilder();
  }

  /**
   * @param dataQualityMetrics the dataQualityMetrics
   * @return InputDatasetInputFacets
   */
  public InputDatasetInputFacets newInputDatasetInputFacets(
      DataQualityMetricsInputDatasetFacet dataQualityMetrics) {
    return new InputDatasetInputFacets(dataQualityMetrics);
  }

  public InputDatasetInputFacetsBuilder newInputDatasetInputFacetsBuilder() {
    return new InputDatasetInputFacetsBuilder();
  }

  public interface BaseFacet {
    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    URI get_producer();

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    URI get_schemaURL();
  }

  public static final class ParentRunFacetRun {
    private final UUID runId;

    /**
     * @param runId The globally unique ID of the run associated with the job.
     */
    @JsonCreator
    private ParentRunFacetRun(@JsonProperty("runId") UUID runId) {
      this.runId = runId;
    }

    /**
     * @return The globally unique ID of the run associated with the job.
     */
    public UUID getRunId() {
      return runId;
    }
  }

  public static final class ParentRunFacetRunBuilder {
    private UUID runId;

    /**
     * @param runId The globally unique ID of the run associated with the job.
     * @return this
     */
    public ParentRunFacetRunBuilder runId(UUID runId) {
      this.runId = runId;
      return this;
    }

    public ParentRunFacetRun build() {
      ParentRunFacetRun __result = new ParentRunFacetRun(runId);
      return __result;
    }
  }

  public static final class DatasourceDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final String name;

    private final URI uri;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param name the name
     * @param uri the uri
     */
    @JsonCreator
    private DatasourceDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("name") String name, @JsonProperty("uri") URI uri) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/DatasourceDatasetFacet");
      this.name = name;
      this.uri = uri;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    public String getName() {
      return name;
    }

    public URI getUri() {
      return uri;
    }
  }

  public final class DatasourceDatasetFacetBuilder {
    private String name;

    private URI uri;

    /**
     * @param name the name
     * @return this
     */
    public DatasourceDatasetFacetBuilder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * @param uri the uri
     * @return this
     */
    public DatasourceDatasetFacetBuilder uri(URI uri) {
      this.uri = uri;
      return this;
    }

    public DatasourceDatasetFacet build() {
      DatasourceDatasetFacet __result = new DatasourceDatasetFacet(OpenLineage.this.producer, name, uri);
      return __result;
    }
  }

  public static final class DocumentationJobFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final String description;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param description The description of the job.
     */
    @JsonCreator
    private DocumentationJobFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("description") String description) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/DocumentationJobFacet");
      this.description = description;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return The description of the job.
     */
    public String getDescription() {
      return description;
    }
  }

  public final class DocumentationJobFacetBuilder {
    private String description;

    /**
     * @param description The description of the job.
     * @return this
     */
    public DocumentationJobFacetBuilder description(String description) {
      this.description = description;
      return this;
    }

    public DocumentationJobFacet build() {
      DocumentationJobFacet __result = new DocumentationJobFacet(OpenLineage.this.producer, description);
      return __result;
    }
  }

  public static final class OutputStatisticsOutputDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final Long rowCount;

    private final Long size;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param rowCount The number of rows written to the dataset
     * @param size The size in bytes written to the dataset
     */
    @JsonCreator
    private OutputStatisticsOutputDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("rowCount") Long rowCount, @JsonProperty("size") Long size) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/OutputStatisticsOutputDatasetFacet");
      this.rowCount = rowCount;
      this.size = size;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return The number of rows written to the dataset
     */
    public Long getRowCount() {
      return rowCount;
    }

    /**
     * @return The size in bytes written to the dataset
     */
    public Long getSize() {
      return size;
    }
  }

  public final class OutputStatisticsOutputDatasetFacetBuilder {
    private Long rowCount;

    private Long size;

    /**
     * @param rowCount The number of rows written to the dataset
     * @return this
     */
    public OutputStatisticsOutputDatasetFacetBuilder rowCount(Long rowCount) {
      this.rowCount = rowCount;
      return this;
    }

    /**
     * @param size The size in bytes written to the dataset
     * @return this
     */
    public OutputStatisticsOutputDatasetFacetBuilder size(Long size) {
      this.size = size;
      return this;
    }

    public OutputStatisticsOutputDatasetFacet build() {
      OutputStatisticsOutputDatasetFacet __result = new OutputStatisticsOutputDatasetFacet(OpenLineage.this.producer, rowCount, size);
      return __result;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final Long rowCount;

    private final Long bytes;

    private final DataQualityMetricsInputDatasetFacetColumnMetrics columnMetrics;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param rowCount The number of rows evaluated
     * @param bytes The size in bytes
     * @param columnMetrics The property key is the column name
     */
    @JsonCreator
    private DataQualityMetricsInputDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("rowCount") Long rowCount, @JsonProperty("bytes") Long bytes,
        @JsonProperty("columnMetrics") DataQualityMetricsInputDatasetFacetColumnMetrics columnMetrics) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/DataQualityMetricsInputDatasetFacet");
      this.rowCount = rowCount;
      this.bytes = bytes;
      this.columnMetrics = columnMetrics;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return The number of rows evaluated
     */
    public Long getRowCount() {
      return rowCount;
    }

    /**
     * @return The size in bytes
     */
    public Long getBytes() {
      return bytes;
    }

    /**
     * @return The property key is the column name
     */
    public DataQualityMetricsInputDatasetFacetColumnMetrics getColumnMetrics() {
      return columnMetrics;
    }
  }

  public final class DataQualityMetricsInputDatasetFacetBuilder {
    private Long rowCount;

    private Long bytes;

    private DataQualityMetricsInputDatasetFacetColumnMetrics columnMetrics;

    /**
     * @param rowCount The number of rows evaluated
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetBuilder rowCount(Long rowCount) {
      this.rowCount = rowCount;
      return this;
    }

    /**
     * @param bytes The size in bytes
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetBuilder bytes(Long bytes) {
      this.bytes = bytes;
      return this;
    }

    /**
     * @param columnMetrics The property key is the column name
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetBuilder columnMetrics(
        DataQualityMetricsInputDatasetFacetColumnMetrics columnMetrics) {
      this.columnMetrics = columnMetrics;
      return this;
    }

    public DataQualityMetricsInputDatasetFacet build() {
      DataQualityMetricsInputDatasetFacet __result = new DataQualityMetricsInputDatasetFacet(OpenLineage.this.producer, rowCount, bytes, columnMetrics);
      return __result;
    }
  }

  public static final class RunFacets {
    private final NominalTimeRunFacet nominalTime;

    private final ParentRunFacet parent;

    @JsonAnySetter
    private final Map<String, CustomFacet> additionalProperties;

    /**
     * @param nominalTime the nominalTime
     * @param parent the parent
     */
    @JsonCreator
    private RunFacets(@JsonProperty("nominalTime") NominalTimeRunFacet nominalTime,
        @JsonProperty("parent") ParentRunFacet parent) {
      this.nominalTime = nominalTime;
      this.parent = parent;
      this.additionalProperties = new HashMap<>();
    }

    public NominalTimeRunFacet getNominalTime() {
      return nominalTime;
    }

    public ParentRunFacet getParent() {
      return parent;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, CustomFacet> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class RunFacetsBuilder {
    private NominalTimeRunFacet nominalTime;

    private ParentRunFacet parent;

    private final Map<String, CustomFacet> additionalProperties = new HashMap<>();

    /**
     * @param nominalTime the nominalTime
     * @return this
     */
    public RunFacetsBuilder nominalTime(NominalTimeRunFacet nominalTime) {
      this.nominalTime = nominalTime;
      return this;
    }

    /**
     * @param parent the parent
     * @return this
     */
    public RunFacetsBuilder parent(ParentRunFacet parent) {
      this.parent = parent;
      return this;
    }

    /**
     * @return this
     */
    public RunFacetsBuilder put(String key, CustomFacet value) {
      this.additionalProperties.put(key, value);return this;
    }

    public RunFacets build() {
      RunFacets __result = new RunFacets(nominalTime, parent);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class JobFacets {
    private final DocumentationJobFacet documentation;

    private final SourceCodeLocationJobFacet sourceCodeLocation;

    private final SQLJobFacet sql;

    @JsonAnySetter
    private final Map<String, CustomFacet> additionalProperties;

    /**
     * @param documentation the documentation
     * @param sourceCodeLocation the sourceCodeLocation
     * @param sql the sql
     */
    @JsonCreator
    private JobFacets(@JsonProperty("documentation") DocumentationJobFacet documentation,
        @JsonProperty("sourceCodeLocation") SourceCodeLocationJobFacet sourceCodeLocation,
        @JsonProperty("sql") SQLJobFacet sql) {
      this.documentation = documentation;
      this.sourceCodeLocation = sourceCodeLocation;
      this.sql = sql;
      this.additionalProperties = new HashMap<>();
    }

    public DocumentationJobFacet getDocumentation() {
      return documentation;
    }

    public SourceCodeLocationJobFacet getSourceCodeLocation() {
      return sourceCodeLocation;
    }

    public SQLJobFacet getSql() {
      return sql;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, CustomFacet> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class JobFacetsBuilder {
    private DocumentationJobFacet documentation;

    private SourceCodeLocationJobFacet sourceCodeLocation;

    private SQLJobFacet sql;

    private final Map<String, CustomFacet> additionalProperties = new HashMap<>();

    /**
     * @param documentation the documentation
     * @return this
     */
    public JobFacetsBuilder documentation(DocumentationJobFacet documentation) {
      this.documentation = documentation;
      return this;
    }

    /**
     * @param sourceCodeLocation the sourceCodeLocation
     * @return this
     */
    public JobFacetsBuilder sourceCodeLocation(SourceCodeLocationJobFacet sourceCodeLocation) {
      this.sourceCodeLocation = sourceCodeLocation;
      return this;
    }

    /**
     * @param sql the sql
     * @return this
     */
    public JobFacetsBuilder sql(SQLJobFacet sql) {
      this.sql = sql;
      return this;
    }

    /**
     * @return this
     */
    public JobFacetsBuilder put(String key, CustomFacet value) {
      this.additionalProperties.put(key, value);return this;
    }

    public JobFacets build() {
      JobFacets __result = new JobFacets(documentation, sourceCodeLocation, sql);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class Job {
    private final String namespace;

    private final String name;

    private final JobFacets facets;

    /**
     * @param namespace The namespace containing that job
     * @param name The unique name for that job within that namespace
     * @param facets The job facets.
     */
    @JsonCreator
    private Job(@JsonProperty("namespace") String namespace, @JsonProperty("name") String name,
        @JsonProperty("facets") JobFacets facets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
    }

    /**
     * @return The namespace containing that job
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that job within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The job facets.
     */
    public JobFacets getFacets() {
      return facets;
    }
  }

  public static final class JobBuilder {
    private String namespace;

    private String name;

    private JobFacets facets;

    /**
     * @param namespace The namespace containing that job
     * @return this
     */
    public JobBuilder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    /**
     * @param name The unique name for that job within that namespace
     * @return this
     */
    public JobBuilder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * @param facets The job facets.
     * @return this
     */
    public JobBuilder facets(JobFacets facets) {
      this.facets = facets;
      return this;
    }

    public Job build() {
      Job __result = new Job(namespace, name, facets);
      return __result;
    }
  }

  public static final class ParentRunFacetJob {
    private final String namespace;

    private final String name;

    /**
     * @param namespace The namespace containing that job
     * @param name The unique name for that job within that namespace
     */
    @JsonCreator
    private ParentRunFacetJob(@JsonProperty("namespace") String namespace,
        @JsonProperty("name") String name) {
      this.namespace = namespace;
      this.name = name;
    }

    /**
     * @return The namespace containing that job
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that job within that namespace
     */
    public String getName() {
      return name;
    }
  }

  public static final class ParentRunFacetJobBuilder {
    private String namespace;

    private String name;

    /**
     * @param namespace The namespace containing that job
     * @return this
     */
    public ParentRunFacetJobBuilder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    /**
     * @param name The unique name for that job within that namespace
     * @return this
     */
    public ParentRunFacetJobBuilder name(String name) {
      this.name = name;
      return this;
    }

    public ParentRunFacetJob build() {
      ParentRunFacetJob __result = new ParentRunFacetJob(namespace, name);
      return __result;
    }
  }

  public static final class DocumentationDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final String description;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param description The description of the dataset.
     */
    @JsonCreator
    private DocumentationDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("description") String description) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/DocumentationDatasetFacet");
      this.description = description;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return The description of the dataset.
     */
    public String getDescription() {
      return description;
    }
  }

  public final class DocumentationDatasetFacetBuilder {
    private String description;

    /**
     * @param description The description of the dataset.
     * @return this
     */
    public DocumentationDatasetFacetBuilder description(String description) {
      this.description = description;
      return this;
    }

    public DocumentationDatasetFacet build() {
      DocumentationDatasetFacet __result = new DocumentationDatasetFacet(OpenLineage.this.producer, description);
      return __result;
    }
  }

  public static final class SourceCodeLocationJobFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final String type;

    private final URI url;

    private final String repoUrl;

    private final String path;

    private final String version;

    private final String tag;

    private final String branch;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param type the source control system
     * @param url the full http URL to locate the file
     * @param repoUrl the URL to the repository
     * @param path the path in the repo containing the source files
     * @param version the current version deployed (not a branch name, the actual unique version)
     * @param tag optional tag name
     * @param branch optional branch name
     */
    @JsonCreator
    private SourceCodeLocationJobFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("type") String type, @JsonProperty("url") URI url,
        @JsonProperty("repoUrl") String repoUrl, @JsonProperty("path") String path,
        @JsonProperty("version") String version, @JsonProperty("tag") String tag,
        @JsonProperty("branch") String branch) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/SourceCodeLocationJobFacet");
      this.type = type;
      this.url = url;
      this.repoUrl = repoUrl;
      this.path = path;
      this.version = version;
      this.tag = tag;
      this.branch = branch;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return the source control system
     */
    public String getType() {
      return type;
    }

    /**
     * @return the full http URL to locate the file
     */
    public URI getUrl() {
      return url;
    }

    /**
     * @return the URL to the repository
     */
    public String getRepoUrl() {
      return repoUrl;
    }

    /**
     * @return the path in the repo containing the source files
     */
    public String getPath() {
      return path;
    }

    /**
     * @return the current version deployed (not a branch name, the actual unique version)
     */
    public String getVersion() {
      return version;
    }

    /**
     * @return optional tag name
     */
    public String getTag() {
      return tag;
    }

    /**
     * @return optional branch name
     */
    public String getBranch() {
      return branch;
    }
  }

  public final class SourceCodeLocationJobFacetBuilder {
    private String type;

    private URI url;

    private String repoUrl;

    private String path;

    private String version;

    private String tag;

    private String branch;

    /**
     * @param type the source control system
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder type(String type) {
      this.type = type;
      return this;
    }

    /**
     * @param url the full http URL to locate the file
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder url(URI url) {
      this.url = url;
      return this;
    }

    /**
     * @param repoUrl the URL to the repository
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder repoUrl(String repoUrl) {
      this.repoUrl = repoUrl;
      return this;
    }

    /**
     * @param path the path in the repo containing the source files
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder path(String path) {
      this.path = path;
      return this;
    }

    /**
     * @param version the current version deployed (not a branch name, the actual unique version)
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder version(String version) {
      this.version = version;
      return this;
    }

    /**
     * @param tag optional tag name
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder tag(String tag) {
      this.tag = tag;
      return this;
    }

    /**
     * @param branch optional branch name
     * @return this
     */
    public SourceCodeLocationJobFacetBuilder branch(String branch) {
      this.branch = branch;
      return this;
    }

    public SourceCodeLocationJobFacet build() {
      SourceCodeLocationJobFacet __result = new SourceCodeLocationJobFacet(OpenLineage.this.producer, type, url, repoUrl, path, version, tag, branch);
      return __result;
    }
  }

  public static final class ParentRunFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final ParentRunFacetRun run;

    private final ParentRunFacetJob job;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param run the run
     * @param job the job
     */
    @JsonCreator
    private ParentRunFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("run") ParentRunFacetRun run, @JsonProperty("job") ParentRunFacetJob job) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/ParentRunFacet");
      this.run = run;
      this.job = job;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    public ParentRunFacetRun getRun() {
      return run;
    }

    public ParentRunFacetJob getJob() {
      return job;
    }
  }

  public final class ParentRunFacetBuilder {
    private ParentRunFacetRun run;

    private ParentRunFacetJob job;

    /**
     * @param run the run
     * @return this
     */
    public ParentRunFacetBuilder run(ParentRunFacetRun run) {
      this.run = run;
      return this;
    }

    /**
     * @param job the job
     * @return this
     */
    public ParentRunFacetBuilder job(ParentRunFacetJob job) {
      this.job = job;
      return this;
    }

    public ParentRunFacet build() {
      ParentRunFacet __result = new ParentRunFacet(OpenLineage.this.producer, run, job);
      return __result;
    }
  }

  public static final class OutputDatasetOutputFacets {
    private final OutputStatisticsOutputDatasetFacet outputStatistics;

    @JsonAnySetter
    private final Map<String, CustomFacet> additionalProperties;

    /**
     * @param outputStatistics the outputStatistics
     */
    @JsonCreator
    private OutputDatasetOutputFacets(
        @JsonProperty("outputStatistics") OutputStatisticsOutputDatasetFacet outputStatistics) {
      this.outputStatistics = outputStatistics;
      this.additionalProperties = new HashMap<>();
    }

    public OutputStatisticsOutputDatasetFacet getOutputStatistics() {
      return outputStatistics;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, CustomFacet> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class OutputDatasetOutputFacetsBuilder {
    private OutputStatisticsOutputDatasetFacet outputStatistics;

    private final Map<String, CustomFacet> additionalProperties = new HashMap<>();

    /**
     * @param outputStatistics the outputStatistics
     * @return this
     */
    public OutputDatasetOutputFacetsBuilder outputStatistics(
        OutputStatisticsOutputDatasetFacet outputStatistics) {
      this.outputStatistics = outputStatistics;
      return this;
    }

    /**
     * @return this
     */
    public OutputDatasetOutputFacetsBuilder put(String key, CustomFacet value) {
      this.additionalProperties.put(key, value);return this;
    }

    public OutputDatasetOutputFacets build() {
      OutputDatasetOutputFacets __result = new OutputDatasetOutputFacets(outputStatistics);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class DatasetFacets {
    private final DocumentationDatasetFacet documentation;

    private final SchemaDatasetFacet schema;

    private final DatasourceDatasetFacet dataSource;

    @JsonAnySetter
    private final Map<String, CustomFacet> additionalProperties;

    /**
     * @param documentation the documentation
     * @param schema the schema
     * @param dataSource the dataSource
     */
    @JsonCreator
    private DatasetFacets(@JsonProperty("documentation") DocumentationDatasetFacet documentation,
        @JsonProperty("schema") SchemaDatasetFacet schema,
        @JsonProperty("dataSource") DatasourceDatasetFacet dataSource) {
      this.documentation = documentation;
      this.schema = schema;
      this.dataSource = dataSource;
      this.additionalProperties = new HashMap<>();
    }

    public DocumentationDatasetFacet getDocumentation() {
      return documentation;
    }

    public SchemaDatasetFacet getSchema() {
      return schema;
    }

    public DatasourceDatasetFacet getDataSource() {
      return dataSource;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, CustomFacet> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class DatasetFacetsBuilder {
    private DocumentationDatasetFacet documentation;

    private SchemaDatasetFacet schema;

    private DatasourceDatasetFacet dataSource;

    private final Map<String, CustomFacet> additionalProperties = new HashMap<>();

    /**
     * @param documentation the documentation
     * @return this
     */
    public DatasetFacetsBuilder documentation(DocumentationDatasetFacet documentation) {
      this.documentation = documentation;
      return this;
    }

    /**
     * @param schema the schema
     * @return this
     */
    public DatasetFacetsBuilder schema(SchemaDatasetFacet schema) {
      this.schema = schema;
      return this;
    }

    /**
     * @param dataSource the dataSource
     * @return this
     */
    public DatasetFacetsBuilder dataSource(DatasourceDatasetFacet dataSource) {
      this.dataSource = dataSource;
      return this;
    }

    /**
     * @return this
     */
    public DatasetFacetsBuilder put(String key, CustomFacet value) {
      this.additionalProperties.put(key, value);return this;
    }

    public DatasetFacets build() {
      DatasetFacets __result = new DatasetFacets(documentation, schema, dataSource);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public interface Dataset {
    /**
     * @return The namespace containing that dataset
     */
    String getNamespace();

    /**
     * @return The unique name for that dataset within that namespace
     */
    String getName();

    /**
     * @return The facets for this dataset
     */
    DatasetFacets getFacets();
  }

  public static final class NominalTimeRunFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final ZonedDateTime nominalStartTime;

    private final ZonedDateTime nominalEndTime;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param nominalStartTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal start time (included) of the run. AKA the schedule time
     * @param nominalEndTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal end time (excluded) of the run. (Should be the nominal start time of the next run)
     */
    @JsonCreator
    private NominalTimeRunFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("nominalStartTime") ZonedDateTime nominalStartTime,
        @JsonProperty("nominalEndTime") ZonedDateTime nominalEndTime) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/NominalTimeRunFacet");
      this.nominalStartTime = nominalStartTime;
      this.nominalEndTime = nominalEndTime;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal start time (included) of the run. AKA the schedule time
     */
    public ZonedDateTime getNominalStartTime() {
      return nominalStartTime;
    }

    /**
     * @return An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal end time (excluded) of the run. (Should be the nominal start time of the next run)
     */
    public ZonedDateTime getNominalEndTime() {
      return nominalEndTime;
    }
  }

  public final class NominalTimeRunFacetBuilder {
    private ZonedDateTime nominalStartTime;

    private ZonedDateTime nominalEndTime;

    /**
     * @param nominalStartTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal start time (included) of the run. AKA the schedule time
     * @return this
     */
    public NominalTimeRunFacetBuilder nominalStartTime(ZonedDateTime nominalStartTime) {
      this.nominalStartTime = nominalStartTime;
      return this;
    }

    /**
     * @param nominalEndTime An [ISO-8601](https://en.wikipedia.org/wiki/ISO_8601) timestamp representing the nominal end time (excluded) of the run. (Should be the nominal start time of the next run)
     * @return this
     */
    public NominalTimeRunFacetBuilder nominalEndTime(ZonedDateTime nominalEndTime) {
      this.nominalEndTime = nominalEndTime;
      return this;
    }

    public NominalTimeRunFacet build() {
      NominalTimeRunFacet __result = new NominalTimeRunFacet(OpenLineage.this.producer, nominalStartTime, nominalEndTime);
      return __result;
    }
  }

  public static final class SchemaDatasetFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final List<SchemaDatasetFacetFields> fields;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param fields The fields of the table.
     */
    @JsonCreator
    private SchemaDatasetFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("fields") List<SchemaDatasetFacetFields> fields) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/SchemaDatasetFacet");
      this.fields = fields;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return The fields of the table.
     */
    public List<SchemaDatasetFacetFields> getFields() {
      return fields;
    }
  }

  public final class SchemaDatasetFacetBuilder {
    private List<SchemaDatasetFacetFields> fields;

    /**
     * @param fields The fields of the table.
     * @return this
     */
    public SchemaDatasetFacetBuilder fields(List<SchemaDatasetFacetFields> fields) {
      this.fields = fields;
      return this;
    }

    public SchemaDatasetFacet build() {
      SchemaDatasetFacet __result = new SchemaDatasetFacet(OpenLineage.this.producer, fields);
      return __result;
    }
  }

  public static final class SQLJobFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    private final String query;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     * @param query the query
     */
    @JsonCreator
    private SQLJobFacet(@JsonProperty("_producer") URI _producer,
        @JsonProperty("query") String query) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/SQLJobFacet");
      this.query = query;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    public String getQuery() {
      return query;
    }
  }

  public final class SQLJobFacetBuilder {
    private String query;

    /**
     * @param query the query
     * @return this
     */
    public SQLJobFacetBuilder query(String query) {
      this.query = query;
      return this;
    }

    public SQLJobFacet build() {
      SQLJobFacet __result = new SQLJobFacet(OpenLineage.this.producer, query);
      return __result;
    }
  }

  public static final class Run {
    private final UUID runId;

    private final RunFacets facets;

    /**
     * @param runId The globally unique ID of the run associated with the job.
     * @param facets The run facets.
     */
    @JsonCreator
    private Run(@JsonProperty("runId") UUID runId, @JsonProperty("facets") RunFacets facets) {
      this.runId = runId;
      this.facets = facets;
    }

    /**
     * @return The globally unique ID of the run associated with the job.
     */
    public UUID getRunId() {
      return runId;
    }

    /**
     * @return The run facets.
     */
    public RunFacets getFacets() {
      return facets;
    }
  }

  public static final class RunBuilder {
    private UUID runId;

    private RunFacets facets;

    /**
     * @param runId The globally unique ID of the run associated with the job.
     * @return this
     */
    public RunBuilder runId(UUID runId) {
      this.runId = runId;
      return this;
    }

    /**
     * @param facets The run facets.
     * @return this
     */
    public RunBuilder facets(RunFacets facets) {
      this.facets = facets;
      return this;
    }

    public Run build() {
      Run __result = new Run(runId, facets);
      return __result;
    }
  }

  public static class CustomFacet implements BaseFacet {
    private final URI _producer;

    private final URI _schemaURL;

    @JsonAnySetter
    private final Map<String, Object> additionalProperties;

    /**
     * @param _producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    @JsonCreator
    public CustomFacet(@JsonProperty("_producer") URI _producer) {
      this._producer = _producer;
      this._schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/CustomFacet");
      this.additionalProperties = new HashMap<>();
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI get_producer() {
      return _producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this facet
     */
    public URI get_schemaURL() {
      return _schemaURL;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Object> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public final class CustomFacetBuilder {
    private final Map<String, Object> additionalProperties = new HashMap<>();

    /**
     * @return this
     */
    public CustomFacetBuilder put(String key, Object value) {
      this.additionalProperties.put(key, value);return this;
    }

    public CustomFacet build() {
      CustomFacet __result = new CustomFacet(OpenLineage.this.producer);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetrics {
    @JsonAnySetter
    private final Map<String, DataQualityMetricsInputDatasetFacetColumnMetricsAdditional> additionalProperties;

    @JsonCreator
    private DataQualityMetricsInputDatasetFacetColumnMetrics() {
      this.additionalProperties = new HashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, DataQualityMetricsInputDatasetFacetColumnMetricsAdditional> getAdditionalProperties(
        ) {
      return additionalProperties;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetricsBuilder {
    private final Map<String, DataQualityMetricsInputDatasetFacetColumnMetricsAdditional> additionalProperties = new HashMap<>();

    /**
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsBuilder put(String key,
        DataQualityMetricsInputDatasetFacetColumnMetricsAdditional value) {
      this.additionalProperties.put(key, value);return this;
    }

    public DataQualityMetricsInputDatasetFacetColumnMetrics build() {
      DataQualityMetricsInputDatasetFacetColumnMetrics __result = new DataQualityMetricsInputDatasetFacetColumnMetrics();
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class InputDataset implements Dataset {
    private final String namespace;

    private final String name;

    private final DatasetFacets facets;

    private final InputDatasetInputFacets inputFacets;

    /**
     * @param namespace The namespace containing that dataset
     * @param name The unique name for that dataset within that namespace
     * @param facets The facets for this dataset
     * @param inputFacets The input facets for this dataset.
     */
    @JsonCreator
    private InputDataset(@JsonProperty("namespace") String namespace,
        @JsonProperty("name") String name, @JsonProperty("facets") DatasetFacets facets,
        @JsonProperty("inputFacets") InputDatasetInputFacets inputFacets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
      this.inputFacets = inputFacets;
    }

    /**
     * @return The namespace containing that dataset
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that dataset within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The facets for this dataset
     */
    public DatasetFacets getFacets() {
      return facets;
    }

    /**
     * @return The input facets for this dataset.
     */
    public InputDatasetInputFacets getInputFacets() {
      return inputFacets;
    }
  }

  public static final class InputDatasetBuilder {
    private String namespace;

    private String name;

    private DatasetFacets facets;

    private InputDatasetInputFacets inputFacets;

    /**
     * @param namespace The namespace containing that dataset
     * @return this
     */
    public InputDatasetBuilder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    /**
     * @param name The unique name for that dataset within that namespace
     * @return this
     */
    public InputDatasetBuilder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * @param facets The facets for this dataset
     * @return this
     */
    public InputDatasetBuilder facets(DatasetFacets facets) {
      this.facets = facets;
      return this;
    }

    /**
     * @param inputFacets The input facets for this dataset.
     * @return this
     */
    public InputDatasetBuilder inputFacets(InputDatasetInputFacets inputFacets) {
      this.inputFacets = inputFacets;
      return this;
    }

    public InputDataset build() {
      InputDataset __result = new InputDataset(namespace, name, facets, inputFacets);
      return __result;
    }
  }

  public static final class OutputDataset implements Dataset {
    private final String namespace;

    private final String name;

    private final DatasetFacets facets;

    private final OutputDatasetOutputFacets outputFacets;

    /**
     * @param namespace The namespace containing that dataset
     * @param name The unique name for that dataset within that namespace
     * @param facets The facets for this dataset
     * @param outputFacets The output facets for this dataset
     */
    @JsonCreator
    private OutputDataset(@JsonProperty("namespace") String namespace,
        @JsonProperty("name") String name, @JsonProperty("facets") DatasetFacets facets,
        @JsonProperty("outputFacets") OutputDatasetOutputFacets outputFacets) {
      this.namespace = namespace;
      this.name = name;
      this.facets = facets;
      this.outputFacets = outputFacets;
    }

    /**
     * @return The namespace containing that dataset
     */
    public String getNamespace() {
      return namespace;
    }

    /**
     * @return The unique name for that dataset within that namespace
     */
    public String getName() {
      return name;
    }

    /**
     * @return The facets for this dataset
     */
    public DatasetFacets getFacets() {
      return facets;
    }

    /**
     * @return The output facets for this dataset
     */
    public OutputDatasetOutputFacets getOutputFacets() {
      return outputFacets;
    }
  }

  public static final class OutputDatasetBuilder {
    private String namespace;

    private String name;

    private DatasetFacets facets;

    private OutputDatasetOutputFacets outputFacets;

    /**
     * @param namespace The namespace containing that dataset
     * @return this
     */
    public OutputDatasetBuilder namespace(String namespace) {
      this.namespace = namespace;
      return this;
    }

    /**
     * @param name The unique name for that dataset within that namespace
     * @return this
     */
    public OutputDatasetBuilder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * @param facets The facets for this dataset
     * @return this
     */
    public OutputDatasetBuilder facets(DatasetFacets facets) {
      this.facets = facets;
      return this;
    }

    /**
     * @param outputFacets The output facets for this dataset
     * @return this
     */
    public OutputDatasetBuilder outputFacets(OutputDatasetOutputFacets outputFacets) {
      this.outputFacets = outputFacets;
      return this;
    }

    public OutputDataset build() {
      OutputDataset __result = new OutputDataset(namespace, name, facets, outputFacets);
      return __result;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles {
    @JsonAnySetter
    private final Map<String, Double> additionalProperties;

    @JsonCreator
    private DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles() {
      this.additionalProperties = new HashMap<>();
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, Double> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantilesBuilder {
    private final Map<String, Double> additionalProperties = new HashMap<>();

    /**
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantilesBuilder put(
        String key, Double value) {
      this.additionalProperties.put(key, value);return this;
    }

    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles build() {
      DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles __result = new DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles();
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetricsAdditional {
    private final Long nullCount;

    private final Long distinctCount;

    private final Double sum;

    private final Double count;

    private final Double min;

    private final Double max;

    private final DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles quantiles;

    /**
     * @param nullCount The number of null values in this column for the rows evaluated
     * @param distinctCount The number of distinct values in this column for the rows evaluated
     * @param sum The total sum of values in this column for the rows evaluated
     * @param count The number of values in this column
     * @param min the min
     * @param max the max
     * @param quantiles The property key is the quantile. Examples: 0.1 0.25 0.5 0.75 1
     */
    @JsonCreator
    private DataQualityMetricsInputDatasetFacetColumnMetricsAdditional(
        @JsonProperty("nullCount") Long nullCount,
        @JsonProperty("distinctCount") Long distinctCount, @JsonProperty("sum") Double sum,
        @JsonProperty("count") Double count, @JsonProperty("min") Double min,
        @JsonProperty("max") Double max,
        @JsonProperty("quantiles") DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles quantiles) {
      this.nullCount = nullCount;
      this.distinctCount = distinctCount;
      this.sum = sum;
      this.count = count;
      this.min = min;
      this.max = max;
      this.quantiles = quantiles;
    }

    /**
     * @return The number of null values in this column for the rows evaluated
     */
    public Long getNullCount() {
      return nullCount;
    }

    /**
     * @return The number of distinct values in this column for the rows evaluated
     */
    public Long getDistinctCount() {
      return distinctCount;
    }

    /**
     * @return The total sum of values in this column for the rows evaluated
     */
    public Double getSum() {
      return sum;
    }

    /**
     * @return The number of values in this column
     */
    public Double getCount() {
      return count;
    }

    public Double getMin() {
      return min;
    }

    public Double getMax() {
      return max;
    }

    /**
     * @return The property key is the quantile. Examples: 0.1 0.25 0.5 0.75 1
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles getQuantiles() {
      return quantiles;
    }
  }

  public static final class DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder {
    private Long nullCount;

    private Long distinctCount;

    private Double sum;

    private Double count;

    private Double min;

    private Double max;

    private DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles quantiles;

    /**
     * @param nullCount The number of null values in this column for the rows evaluated
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder nullCount(
        Long nullCount) {
      this.nullCount = nullCount;
      return this;
    }

    /**
     * @param distinctCount The number of distinct values in this column for the rows evaluated
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder distinctCount(
        Long distinctCount) {
      this.distinctCount = distinctCount;
      return this;
    }

    /**
     * @param sum The total sum of values in this column for the rows evaluated
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder sum(Double sum) {
      this.sum = sum;
      return this;
    }

    /**
     * @param count The number of values in this column
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder count(Double count) {
      this.count = count;
      return this;
    }

    /**
     * @param min the min
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder min(Double min) {
      this.min = min;
      return this;
    }

    /**
     * @param max the max
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder max(Double max) {
      this.max = max;
      return this;
    }

    /**
     * @param quantiles The property key is the quantile. Examples: 0.1 0.25 0.5 0.75 1
     * @return this
     */
    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalBuilder quantiles(
        DataQualityMetricsInputDatasetFacetColumnMetricsAdditionalQuantiles quantiles) {
      this.quantiles = quantiles;
      return this;
    }

    public DataQualityMetricsInputDatasetFacetColumnMetricsAdditional build() {
      DataQualityMetricsInputDatasetFacetColumnMetricsAdditional __result = new DataQualityMetricsInputDatasetFacetColumnMetricsAdditional(nullCount, distinctCount, sum, count, min, max, quantiles);
      return __result;
    }
  }

  public static final class RunEvent {
    private final String eventType;

    private final ZonedDateTime eventTime;

    private final Run run;

    private final Job job;

    private final List<InputDataset> inputs;

    private final List<OutputDataset> outputs;

    private final URI producer;

    private final URI schemaURL;

    /**
     * @param eventType the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
     * @param eventTime the time the event occured at
     * @param run the run
     * @param job the job
     * @param inputs The set of **input** datasets.
     * @param outputs The set of **output** datasets.
     * @param producer URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    @JsonCreator
    private RunEvent(@JsonProperty("eventType") String eventType,
        @JsonProperty("eventTime") ZonedDateTime eventTime, @JsonProperty("run") Run run,
        @JsonProperty("job") Job job, @JsonProperty("inputs") List<InputDataset> inputs,
        @JsonProperty("outputs") List<OutputDataset> outputs,
        @JsonProperty("producer") URI producer) {
      this.eventType = eventType;
      this.eventTime = eventTime;
      this.run = run;
      this.job = job;
      this.inputs = inputs;
      this.outputs = outputs;
      this.producer = producer;
      this.schemaURL = URI.create("https://openlineage.io/spec/1-0-0/OpenLineage.json#/definitions/RunEvent");
    }

    /**
     * @return the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
     */
    public String getEventType() {
      return eventType;
    }

    /**
     * @return the time the event occured at
     */
    public ZonedDateTime getEventTime() {
      return eventTime;
    }

    public Run getRun() {
      return run;
    }

    public Job getJob() {
      return job;
    }

    /**
     * @return The set of **input** datasets.
     */
    public List<InputDataset> getInputs() {
      return inputs;
    }

    /**
     * @return The set of **output** datasets.
     */
    public List<OutputDataset> getOutputs() {
      return outputs;
    }

    /**
     * @return URI identifying the producer of this metadata. For example this could be a git url with a given tag or sha
     */
    public URI getProducer() {
      return producer;
    }

    /**
     * @return The JSON Pointer (https://tools.ietf.org/html/rfc6901) URL to the corresponding version of the schema definition for this RunEvent
     */
    public URI getSchemaURL() {
      return schemaURL;
    }
  }

  public final class RunEventBuilder {
    private String eventType;

    private ZonedDateTime eventTime;

    private Run run;

    private Job job;

    private List<InputDataset> inputs;

    private List<OutputDataset> outputs;

    /**
     * @param eventType the current transition of the run state. It is required to issue 1 START event and 1 of [ COMPLETE, ABORT, FAIL ] event per run. Additional events with OTHER eventType can be added to the same run. For example to send additional metadata after the run is complete
     * @return this
     */
    public RunEventBuilder eventType(String eventType) {
      this.eventType = eventType;
      return this;
    }

    /**
     * @param eventTime the time the event occured at
     * @return this
     */
    public RunEventBuilder eventTime(ZonedDateTime eventTime) {
      this.eventTime = eventTime;
      return this;
    }

    /**
     * @param run the run
     * @return this
     */
    public RunEventBuilder run(Run run) {
      this.run = run;
      return this;
    }

    /**
     * @param job the job
     * @return this
     */
    public RunEventBuilder job(Job job) {
      this.job = job;
      return this;
    }

    /**
     * @param inputs The set of **input** datasets.
     * @return this
     */
    public RunEventBuilder inputs(List<InputDataset> inputs) {
      this.inputs = inputs;
      return this;
    }

    /**
     * @param outputs The set of **output** datasets.
     * @return this
     */
    public RunEventBuilder outputs(List<OutputDataset> outputs) {
      this.outputs = outputs;
      return this;
    }

    public RunEvent build() {
      RunEvent __result = new RunEvent(eventType, eventTime, run, job, inputs, outputs, OpenLineage.this.producer);
      return __result;
    }
  }

  public static final class SchemaDatasetFacetFields {
    private final String name;

    private final String type;

    private final String description;

    /**
     * @param name The name of the field.
     * @param type The type of the field.
     * @param description The description of the field.
     */
    @JsonCreator
    private SchemaDatasetFacetFields(@JsonProperty("name") String name,
        @JsonProperty("type") String type, @JsonProperty("description") String description) {
      this.name = name;
      this.type = type;
      this.description = description;
    }

    /**
     * @return The name of the field.
     */
    public String getName() {
      return name;
    }

    /**
     * @return The type of the field.
     */
    public String getType() {
      return type;
    }

    /**
     * @return The description of the field.
     */
    public String getDescription() {
      return description;
    }
  }

  public static final class SchemaDatasetFacetFieldsBuilder {
    private String name;

    private String type;

    private String description;

    /**
     * @param name The name of the field.
     * @return this
     */
    public SchemaDatasetFacetFieldsBuilder name(String name) {
      this.name = name;
      return this;
    }

    /**
     * @param type The type of the field.
     * @return this
     */
    public SchemaDatasetFacetFieldsBuilder type(String type) {
      this.type = type;
      return this;
    }

    /**
     * @param description The description of the field.
     * @return this
     */
    public SchemaDatasetFacetFieldsBuilder description(String description) {
      this.description = description;
      return this;
    }

    public SchemaDatasetFacetFields build() {
      SchemaDatasetFacetFields __result = new SchemaDatasetFacetFields(name, type, description);
      return __result;
    }
  }

  public static final class InputDatasetInputFacets {
    private final DataQualityMetricsInputDatasetFacet dataQualityMetrics;

    @JsonAnySetter
    private final Map<String, CustomFacet> additionalProperties;

    /**
     * @param dataQualityMetrics the dataQualityMetrics
     */
    @JsonCreator
    private InputDatasetInputFacets(
        @JsonProperty("dataQualityMetrics") DataQualityMetricsInputDatasetFacet dataQualityMetrics) {
      this.dataQualityMetrics = dataQualityMetrics;
      this.additionalProperties = new HashMap<>();
    }

    public DataQualityMetricsInputDatasetFacet getDataQualityMetrics() {
      return dataQualityMetrics;
    }

    /**
     * @return additional properties
     */
    @JsonAnyGetter
    public Map<String, CustomFacet> getAdditionalProperties() {
      return additionalProperties;
    }
  }

  public static final class InputDatasetInputFacetsBuilder {
    private DataQualityMetricsInputDatasetFacet dataQualityMetrics;

    private final Map<String, CustomFacet> additionalProperties = new HashMap<>();

    /**
     * @param dataQualityMetrics the dataQualityMetrics
     * @return this
     */
    public InputDatasetInputFacetsBuilder dataQualityMetrics(
        DataQualityMetricsInputDatasetFacet dataQualityMetrics) {
      this.dataQualityMetrics = dataQualityMetrics;
      return this;
    }

    /**
     * @return this
     */
    public InputDatasetInputFacetsBuilder put(String key, CustomFacet value) {
      this.additionalProperties.put(key, value);return this;
    }

    public InputDatasetInputFacets build() {
      InputDatasetInputFacets __result = new InputDatasetInputFacets(dataQualityMetrics);
      __result.getAdditionalProperties().putAll(additionalProperties);
      return __result;
    }
  }
}
