/*
 * Copyright 2015 David Russell
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.onetapbeyond.renjin.r.executor.tasks;

import io.onetapbeyond.renjin.r.executor.*;
import io.onetapbeyond.renjin.r.executor.manager.RenjinManager;
import io.onetapbeyond.renjin.r.executor.results.RenjinResultImpl;
import java.io.*;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

import javax.script.*;
import org.renjin.sexp.*;
import org.renjin.eval.Context;
import org.renjin.primitives.io.serialization.*;

/*
 * Concrete implementation of Renjin executor task.
 * 
 * Instances of {@link RenjinTask} as generated by
 * the {@link Renjin} task builder.
 */
public class RenjinTaskImpl implements RenjinTask {

	/*
	 * Generated SUID for RenjinTask, generated using serialver. 
	 */
	private static final long serialVersionUID = -2524222643305532748L;

	/*
	 * RenjinTask R code.
	 */
	private final String code;

	/*
	 * RenjinTask SEXP input data. If disableTaskSerialization
	 * is false, data is in serialized form, otherwise data
	 * is in native SEXP encoded form.
	 */
	private final Map<String,Object> sexpInputs;

	/*
	 * RenjinTask primitive inputs.
	 */
	private final Map<String,Object> primInputs;

	/*
	 * By default, all data maintained on RenjinTask are serialized.
	 */
	private boolean disableTaskSerialization;

	/*
	 * By default, all data maintained on RenjinResult are serialized.
	 */
	private boolean disableResultSerialization;

	/*
	 * Dedicated Renjin ScriptEngine for task, optional.
	 */
	private transient ScriptEngine suppliedEngine;
	
	/*
	 * If suppliedEngine != null and enabled, auto clear bindings and
	 * workspace on suppliedEngine following task execution. This
	 * should be enabled when "stateless" usage of the suppliedEngine
	 * is required, otherwise suppliedEngine is "stateful".
	 */
	private transient boolean autoClearSuppliedEngine;

	/*
	 * Reference to the ScriptEngine used to execute the task itself.
	 */
	private transient ScriptEngine liveEngine;

	public RenjinTaskImpl(String code,
						  Map<String,Object> sexpInputs,
						  Map<String,Object> primInputs,
						  boolean disableTaskSerialization,
						  boolean disableResultSerialization,
						  ScriptEngine suppliedEngine,
						  boolean autoClearSuppliedEngine) {

		this.code = code;
		this.sexpInputs = sexpInputs;
		this.primInputs = primInputs;
		this.disableTaskSerialization = disableTaskSerialization;
		this.disableResultSerialization = disableResultSerialization;
		this.suppliedEngine = suppliedEngine;
		this.autoClearSuppliedEngine = autoClearSuppliedEngine;
		this.liveEngine = null;
	}

	public RenjinResult execute() {

		RenjinResult result = null;

		try {

			long taskStart = System.currentTimeMillis();

			/*
			 * Use Renjin ScriptEngine provided by external application,
			 * otherwise retrieve a Renjin ScriptEngine instance from the
			 * RenjinManager.
			 */
			liveEngine = (suppliedEngine != null) ? suppliedEngine :
								 RenjinManager.POOL.getEngine();

			/*
			 * Set SEXP data inputs on the Renjin ScriptEngine.
			 */
			if(disableTaskSerialization) {
				// Process unserialized SEXP data inputs.
				for (Map.Entry<String, Object> pair : sexpInputs.entrySet()) {
					liveEngine.put(pair.getKey(), pair.getValue());
				}
			} else {
				// Process serialized SEXP data inputs.
				for (Map.Entry<String, Object> pair : sexpInputs.entrySet()) {
					liveEngine.put(pair.getKey(),
						deserializeSEXP((byte[]) pair.getValue()));
				}
			}

			/*
			 * Set primitive data inputs on the Renjin ScriptEngine.
			 */
			for (Map.Entry<String, Object> pair : primInputs.entrySet()) {
				liveEngine.put(pair.getKey(), pair.getValue());
			}

			/*
			 * Execute code on the RenjinTask.
			 */
			long execstart = System.currentTimeMillis();
			SEXP output = (SEXP) liveEngine.eval(code);
			long execend = System.currentTimeMillis();

			if(disableResultSerialization) {

				/*
				 * Set SEXP encoded data on RenjinResult.
				 */
				result = new RenjinResultImpl(buildResultMap(output,
								(System.currentTimeMillis()-taskStart)));
			} else {

				/*
				 * Set SEXP serialized data on RenjinResult.
				 */
				result =
					new RenjinResultImpl(buildResultMap(serializeSEXP(output),
								(System.currentTimeMillis()-taskStart)));
			}

		} catch(Throwable cause) {

			result = new RenjinResultImpl(buildResultMap(cause.getMessage(),
																	cause));

		} finally {

			/*
			 * Clear engine bindings and workspace if engineFromPool
			 * or if engineSuppliedAndStateless.
			 */
			try {
				if(engineFromPool() || engineSuppliedAndStateless()) {
					liveEngine.getBindings(ScriptContext.ENGINE_SCOPE).clear();
					liveEngine.eval(RENJIN_CLEAR);
				}
			} catch(ScriptException sex) {}
			/*
			 * If engineFromPool, release engine back to the pool.
			 */
			if(engineFromPool()) {
				RenjinManager.POOL.releaseEngine(liveEngine);
			}
		}

		return result;
	}

	private boolean engineFromPool() {
		return (liveEngine != suppliedEngine);
	}

	private boolean engineSuppliedAndStateless() {
		return (autoClearSuppliedEngine && !engineFromPool());
	}

	/* 
	 * Build resultMap based on successful task execution data 
	 * and meta-data to support {@link RenjinResult}.
	 */
	private Map<String,Object> buildResultMap(Object output,
											  long timeTaken) {

		Map<String,Object> resultMap = new HashMap();
		resultMap.put("success", true);
		resultMap.put("timeTaken", timeTaken);
		resultMap.put("output", output);
		resultMap.put("outputSerialized", !disableResultSerialization);
		resultMap.putAll(buildInputMap());
		resultMap.put("error", null);
		resultMap.put("cause", null);
		return resultMap;
	}

	/* 
	 * Build resultMap based on failed task execution data 
	 * and meta-data to support {@link RenjinResult}.
	 */
	private Map<String,Object> buildResultMap(String error,
											  Throwable cause) {

		Map<String,Object> resultMap = new HashMap();
		resultMap.put("success", false);
		resultMap.put("timeTaken", 0L);
		resultMap.put("output", null);
		resultMap.put("outputSerialized", !disableResultSerialization);
		resultMap.putAll(buildInputMap());
		resultMap.put("error", error);
		resultMap.put("cause", cause);
		return resultMap;
	}

	/* 
	 * Build inputMap data and meta-data based on
	 * {@link RenjinTask} inputs and serialization options.
	 */
	private Map<String,Object> buildInputMap() {
		Map<String,Object> inputMap = new HashMap();
		inputMap.put("sexpInputs", sexpInputs);
		inputMap.put("primInputs", primInputs);
		inputMap.put("inputSerialized", !disableTaskSerialization);
		return inputMap;
	}

	private static final Context TLC = Context.newTopLevelContext();

	public static SEXP deserializeSEXP(byte[] data) {

		SEXP deserializedSEXP = null;

		try {
			long start = System.currentTimeMillis();
			ByteArrayInputStream bais = new
				        	ByteArrayInputStream(data);
		    RDataReader reader = new RDataReader(TLC, bais);
		    deserializedSEXP = reader.readFile();
		} catch(Exception dex) {}

		return deserializedSEXP;
	}

	public static byte[] serializeSEXP(SEXP data) {

		byte[] serializedSEXP = null;

		try {
			long start = System.currentTimeMillis();
	        ByteArrayOutputStream baos =
			        	new ByteArrayOutputStream();
	        RDataWriter writer = new RDataWriter(TLC, baos);
		    writer.save(data);
			serializedSEXP = baos.toByteArray();

		} catch(Exception dex) {}

		return serializedSEXP;
	}

	/*
	 * Renjin R command to 'refresh' ScriptEngine after use.
	 */
	private transient final String RENJIN_CLEAR = "rm(list = ls())";


}
