/*
 * Copyright 2015 David Russell
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.onetapbeyond.renjin.r.executor.tasks;

import io.onetapbeyond.renjin.r.executor.*;
import io.onetapbeyond.renjin.r.executor.manager.RenjinManager;
import io.onetapbeyond.renjin.r.executor.results.RenjinResultImpl;
import java.io.*;
import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

import javax.script.*;
import org.renjin.sexp.*;
import org.renjin.eval.Context;
import org.renjin.primitives.io.serialization.*;

/*
 * Concrete implementation of Renjin executor task.
 * 
 * Instances of {@link RenjinTask} as generated by
 * the {@link Renjin} task builder.
 */
public class RenjinTaskImpl implements RenjinTask {

	/*
	 * Generated SUID for RenjinTask, generated using serialver. 
	 */
	private static final long serialVersionUID = -2524222643305532748L;

	/*
	 * RenjinTask R code.
	 */
	private final String code;

	/*
	 * RenjinTask SEXP input data. If serializableTasks is 
	 * enabled, data is in serialized form, otherwise data
	 * is in native SEXP encoded form.
	 */
	private final Map<String,Object> sexpInputs;

	/*
	 * RenjinTask primitive inputs.
	 */
	private final Map<String,Object> primInputs;

	/*
	 * When enabled, all data maintained on RenjinTask are serializable.
	 */
	private boolean serializableTasks;

	/*
	 * When enabled, all data maintained on RenjinTask are serializable.
	 */
	private boolean serializableResults;

	/*
	 * Dedicated Renjin ScriptEngine for task, optional.
	 */
	private transient ScriptEngine suppliedEngine;
	
	/*
	 * If suppliedEngine != null and enabled, auto clear bindings and
	 * workspace on suppliedEngine following task execution. This
	 * should be enabled when "stateless" usage of the suppliedEngine
	 * is required, otherwise suppliedEngine is "stateful".
	 */
	private transient boolean autoClearSuppliedEngine;

	/*
	 * Reference to the ScriptEngine used to execute the task itself.
	 */
	private transient ScriptEngine liveEngine;

	public RenjinTaskImpl(String code,
						  Map<String,Object> sexpInputs,
						  Map<String,Object> primInputs,
						  boolean serializableTasks,
						  boolean serializableResults,
						  ScriptEngine suppliedEngine,
						  boolean autoClearSuppliedEngine) {

		this.code = code;
		this.sexpInputs = sexpInputs;
		this.primInputs = primInputs;
		this.serializableTasks = serializableTasks;
		this.serializableResults = serializableResults;
		this.suppliedEngine = suppliedEngine;
		this.autoClearSuppliedEngine = autoClearSuppliedEngine;
		this.liveEngine = null;
	}

	public RenjinResult execute() {

		RenjinResult result = null;

		try {

			long taskStart = System.currentTimeMillis();

			/*
			 * Use Renjin ScriptEngine provided by external application,
			 * otherwise retrieve a Renjin ScriptEngine instance from the
			 * RenjinManager.
			 */
			liveEngine = (suppliedEngine != null) ? suppliedEngine :
								 RenjinManager.POOL.getEngine();

			/*
			 * Set SEXP data inputs on the Renjin ScriptEngine.
			 */
			if(serializableTasks) {
				// Process serialized SEXP data inputs.
				for (Map.Entry<String, Object> pair : sexpInputs.entrySet()) {
					liveEngine.put(pair.getKey(),
						deserializeInput(pair.getKey(),
									(byte[]) pair.getValue()));
				}

			} else {
				// Process unserialized SEXP data inputs.
				for (Map.Entry<String, Object> pair : sexpInputs.entrySet()) {
					liveEngine.put(pair.getKey(), pair.getValue());
				}

			}

			/*
			 * Set primitive data inputs on the Renjin ScriptEngine.
			 */
			for (Map.Entry<String, Object> pair : primInputs.entrySet()) {
				liveEngine.put(pair.getKey(), pair.getValue());
			}

			/*
			 * Execute code on the RenjinTask.
			 */
			long execstart = System.currentTimeMillis();
			SEXP output = (SEXP) liveEngine.eval(code);
			long execend = System.currentTimeMillis();

			if(serializableResults) {

				/*
				 * Set serialized data on RenjinResult.
				 */
				result = new RenjinResultImpl(serializeInput("result", output),
							(System.currentTimeMillis()-taskStart));

			} else {

				/*
				 * Set SEXP encoded data on RenjinResult.
				 */
				result = new RenjinResultImpl(output,
							(System.currentTimeMillis()-taskStart));

			}

		} catch(Throwable cause) {

			result = new RenjinResultImpl(cause.getMessage(), cause);

		} finally {

			/*
			 * Clear engine bindings and workspace if engineFromPool
			 * or if engineSuppliedAndStateless.
			 */
			try {
				if(engineFromPool() || engineSuppliedAndStateless()) {
					liveEngine.getBindings(ScriptContext.ENGINE_SCOPE).clear();
					liveEngine.eval(RENJIN_CLEAR);
				}
			} catch(ScriptException sex) {}
			/*
			 * If engineFromPool, release engine back to the pool.
			 */
			if(engineFromPool()) {
				RenjinManager.POOL.releaseEngine(liveEngine);
			}
		}

		return result;
	}

	private boolean engineFromPool() {
		return (liveEngine != suppliedEngine);
	}

	private boolean engineSuppliedAndStateless() {
		return (autoClearSuppliedEngine && !engineFromPool());
	}

	private static final Context TLC = Context.newTopLevelContext();

	public static SEXP deserializeInput(String name, byte[] inputData) {

		SEXP deserializedInput = null;

		try {
			long start = System.currentTimeMillis();
			ByteArrayInputStream bais = new
				        	ByteArrayInputStream(inputData);
		    RDataReader reader = new RDataReader(TLC, bais);
		    deserializedInput = reader.readFile();
		} catch(Exception dex) {}

		return deserializedInput;

	}

	public static byte[] serializeInput(String name, SEXP inputData) {

		byte[] serializedInput = null;

		try {
			long start = System.currentTimeMillis();
	        ByteArrayOutputStream baos =
			        	new ByteArrayOutputStream();
	        RDataWriter writer = new RDataWriter(TLC, baos);
		    writer.save(inputData);
			serializedInput = baos.toByteArray();

		} catch(Exception dex) {}

		return serializedInput;
	}

	/*
	 * Renjin R command to 'refresh' ScriptEngine after use.
	 */
	private transient final String RENJIN_CLEAR = "rm(list = ls())";


}
