package io.nuov.spring.error;

import io.nuov.error.*;
import io.nuov.logging.LoggingDependency;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.NoHandlerFoundException;

/** The class AbstractResponseEntityExceptionHandler. */
public abstract class AbstractResponseEntityExceptionHandler
    extends org.springframework.web.servlet.mvc.method.annotation.ResponseEntityExceptionHandler
    implements LoggingDependency {

  /** The logger. */
  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  @Override
  public Logger getLogger() {
    return this.logger;
  }

  /**
   * Handles all other Exceptions.
   *
   * @param exception the exception
   * @param webRequest the web request
   * @return the response entity
   */
  @ExceptionHandler({Exception.class})
  public ResponseEntity<Object> handleAll(final Exception exception, final WebRequest webRequest) {

    // log exception
    this.logException(exception, webRequest);

    // report exception
    this.reportException(exception, webRequest);

    // respond
    return new ExceptionResponseEntity(
        "An internal service error has occurred.", HttpStatus.INTERNAL_SERVER_ERROR);
  }

  /**
   * Handles BadRequestException.
   *
   * @param badRequestException the BadRequestException
   * @return the response entity
   */
  @ExceptionHandler(BadRequestException.class)
  protected ResponseEntity<Object> handleBadRequest(final BadRequestException badRequestException) {
    return new ExceptionResponseEntity(badRequestException, HttpStatus.BAD_REQUEST);
  }

  /**
   * Handles ForbiddenException.
   *
   * @param forbiddenException the ForbiddenException
   * @return the response entity
   */
  @ExceptionHandler(ForbiddenException.class)
  protected ResponseEntity<Object> handleForbidden(final ForbiddenException forbiddenException) {
    return new ExceptionResponseEntity(forbiddenException, HttpStatus.FORBIDDEN);
  }

  @Override
  protected ResponseEntity<Object> handleHttpMediaTypeNotSupported(
      final HttpMediaTypeNotSupportedException httpMediaTypeNotSupportedException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message =
        httpMediaTypeNotSupportedException.getContentType() + " media type is not supported.";
    return new ExceptionResponseEntity(message, HttpStatus.UNSUPPORTED_MEDIA_TYPE);
  }

  @Override
  protected ResponseEntity<Object> handleHttpMessageNotReadable(
      final HttpMessageNotReadableException httpMessageNotReadableException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message = "Malformed JSON request";
    return new ExceptionResponseEntity(message, HttpStatus.BAD_REQUEST);
  }

  @Override
  protected ResponseEntity<Object> handleHttpMessageNotWritable(
      final HttpMessageNotWritableException httpMessageNotWritableException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message = "Error writing JSON output";
    return new ExceptionResponseEntity(message, HttpStatus.INTERNAL_SERVER_ERROR);
  }

  @Override
  protected ResponseEntity<Object> handleHttpRequestMethodNotSupported(
      final HttpRequestMethodNotSupportedException httpRequestMethodNotSupportedException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message = "This request is not supported.";
    return new ExceptionResponseEntity(message, HttpStatus.BAD_REQUEST);
  }

  /**
   * Handles LockedException.
   *
   * @param lockedException the LockedException
   * @return the response entity
   */
  @ExceptionHandler(LockedException.class)
  protected ResponseEntity<Object> handleLocked(final LockedException lockedException) {
    return new ExceptionResponseEntity(lockedException, HttpStatus.LOCKED);
  }

  @Override
  protected ResponseEntity<Object> handleMissingServletRequestParameter(
      final MissingServletRequestParameterException missingServletRequestParameterException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message =
        missingServletRequestParameterException.getParameterName() + " parameter is missing";
    return new ExceptionResponseEntity(message, HttpStatus.BAD_REQUEST);
  }

  @Override
  protected ResponseEntity<Object> handleNoHandlerFoundException(
      final NoHandlerFoundException noHandlerFoundException,
      final HttpHeaders httpHeaders,
      final HttpStatusCode httpStatusCode,
      final WebRequest webRequest) {
    final String message =
        String.format(
            "Could not find the %s method for URL %s",
            noHandlerFoundException.getHttpMethod(), noHandlerFoundException.getRequestURL());
    return new ExceptionResponseEntity(message, HttpStatus.BAD_REQUEST);
  }

  /**
   * Handles NotFoundException.
   *
   * @param notFoundException the NotFoundException
   * @return the response entity
   */
  @ExceptionHandler(NotFoundException.class)
  protected ResponseEntity<Object> handleNotFound(final NotFoundException notFoundException) {
    return new ExceptionResponseEntity(notFoundException, HttpStatus.NOT_FOUND);
  }

  /**
   * Handles NotModifiedException.
   *
   * @param notModifiedException the NotModifiedException
   * @return the response entity
   */
  @ExceptionHandler(NotModifiedException.class)
  protected ResponseEntity<Object> handleNotModified(
      final NotModifiedException notModifiedException) {
    return new ExceptionResponseEntity(notModifiedException, HttpStatus.NOT_MODIFIED);
  }

  /**
   * Handles ServiceUnavailableException.
   *
   * @param serviceUnavailableException the ServiceUnavailableException
   * @return the response entity
   */
  @ExceptionHandler(ServiceUnavailableException.class)
  protected ResponseEntity<Object> handleServiceUnavailable(
      final ServiceUnavailableException serviceUnavailableException) {
    return new ExceptionResponseEntity(serviceUnavailableException, HttpStatus.SERVICE_UNAVAILABLE);
  }

  /**
   * Handles UnauthorizedException.
   *
   * @param unauthorizedException the UnauthorizedException
   * @return the response entity
   */
  @ExceptionHandler(UnauthorizedException.class)
  protected ResponseEntity<Object> handleUnauthorized(
      final UnauthorizedException unauthorizedException) {
    return new ExceptionResponseEntity(unauthorizedException, HttpStatus.UNAUTHORIZED);
  }

  /**
   * Logs the exception.
   *
   * <p>This method exists to be overridden.
   *
   * @param exception the exception
   * @param webRequest the web request
   */
  protected void logException(final Exception exception, final WebRequest webRequest) {
    this.getLogger().error(exception.getMessage(), exception);
  }

  /**
   * Reports the exception.
   *
   * @param exception the exception
   * @param webRequest the web request
   */
  protected abstract void reportException(final Exception exception, final WebRequest webRequest);
}
