package io.nuov.firebase.authentication;

import com.google.firebase.auth.*;
import io.nuov.firebase.initializer.FirebaseInitializer;
import io.nuov.validator.EmailValidator;
import io.nuov.validator.Nouns;

import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutionException;

/** The class FirebaseAuthentication. */
@SuppressWarnings("unused")
public class FirebaseAuthentication {

  /** Instantiates a FirebaseAuthentication. */
  public FirebaseAuthentication() {
    FirebaseInitializer.initialize();
  }

  /**
   * Adds a User.
   *
   * @param email the email address
   * @return the uid from the user
   */
  public String addUser(final String email) {

    // validate
    EmailValidator.the(Nouns.ARGUMENT, "email", email).validate();

    // assign
    UserRecord.CreateRequest createRequest =
        new UserRecord.CreateRequest()
            .setEmail(email)
            .setEmailVerified(false)
            .setPassword(UUID.randomUUID().toString())
            .setDisabled(false);

    UserRecord userRecord;
    try {
      userRecord = FirebaseAuth.getInstance().createUser(createRequest);
    } catch (final FirebaseAuthException firebaseAuthException) {
      throw new RuntimeException(firebaseAuthException);
    }

    return userRecord.getUid();
  }

  /**
   * Gets the password reset link.
   *
   * @param email the email address
   * @param url the (redirect) url
   * @return the password reset link
   */
  public String getPasswordResetLink(final String email, final String url) {

    // validate
    EmailValidator.the(Nouns.ARGUMENT, "email", email).validate();

    // account settings
    ActionCodeSettings actionCodeSettings = ActionCodeSettings.builder().setUrl(url).build();

    try {
      return FirebaseAuth.getInstance().generatePasswordResetLink(email, actionCodeSettings);
    } catch (final FirebaseAuthException firebaseAuthException) {
      throw new RuntimeException(firebaseAuthException);
    }
  }

  /**
   * Gets the uid.
   *
   * @param email the email address
   * @return the uid
   */
  public String getUid(final String email) {

    // validate
    EmailValidator.the(Nouns.ARGUMENT, "email", email).validate();

    try {

      // get the user
      UserRecord userRecord = FirebaseAuth.getInstance().getUserByEmail(email);
      return userRecord.getUid();

    } catch (final FirebaseAuthException firebaseAuthException) {
      throw new RuntimeException(firebaseAuthException);
    }
  }

  /**
   * Checks if a user exists.
   *
   * @param email the email address
   * @return true if the user exists
   */
  public boolean userExists(final String email) {

    // validate
    EmailValidator.the(Nouns.ARGUMENT, "email", email).validate();

    try {
      GetUsersResult getUsersResult =
          FirebaseAuth.getInstance().getUsersAsync(List.of(new EmailIdentifier(email))).get();
      return getUsersResult.getUsers().size() == 1;
    }

    // exceptions
    catch (final InterruptedException | ExecutionException exception) {
      throw new RuntimeException(exception);
    }
  }
}
