/*
 * Copyright 2018 NEM
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.nem.core.crypto;

/**
 * Wraps IES encryption and decryption logic.
 */
public class Cipher implements BlockCipher {
    private final BlockCipher cipher;

    /**
     * Creates a cipher around a sender KeyPair and recipient KeyPair.
     *
     * @param senderKeyPair    The sender KeyPair. The sender's private key is required for encryption.
     * @param recipientKeyPair The recipient KeyPair. The recipient's private key is required for decryption.
     */
    public Cipher(final KeyPair senderKeyPair, final KeyPair recipientKeyPair) {
        this(senderKeyPair, recipientKeyPair, CryptoEngines.defaultEngine());
    }

    /**
     * Creates a cipher around a sender KeyPair and recipient KeyPair.
     *
     * @param senderKeyPair    The sender KeyPair. The sender's private key is required for encryption.
     * @param recipientKeyPair The recipient KeyPair. The recipient's private key is required for decryption.
     * @param engine           The crypto engine.
     */
    public Cipher(final KeyPair senderKeyPair, final KeyPair recipientKeyPair, final CryptoEngine engine) {
        this(engine.createBlockCipher(senderKeyPair, recipientKeyPair));
    }

    /**
     * Creates a cipher around a cipher.
     *
     * @param cipher The cipher.
     */
    public Cipher(final BlockCipher cipher) {
        this.cipher = cipher;
    }

    @Override
    public byte[] encrypt(final byte[] input) {
        return this.cipher.encrypt(input);
    }

    @Override
    public byte[] decrypt(final byte[] input) {
        return this.cipher.decrypt(input);
    }
}
