/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Binary layout for a mosaic expiry receipt. */
public final class MosaicExpiryReceiptBuilder extends ReceiptBuilder implements Serializer {
    /** Expiring mosaic id. */
    private final MosaicIdDto artifactId;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MosaicExpiryReceiptBuilder(final DataInputStream stream) {
        super(stream);
        this.artifactId = MosaicIdDto.loadFromBinary(stream);
    }

    /**
     * Constructor.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring mosaic id.
     */
    protected MosaicExpiryReceiptBuilder(final short version, final ReceiptTypeDto type, final MosaicIdDto artifactId) {
        super(version, type);
        GeneratorUtils.notNull(artifactId, "artifactId is null");
        this.artifactId = artifactId;
    }

    /**
     * Creates an instance of MosaicExpiryReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param artifactId Expiring mosaic id.
     * @return Instance of MosaicExpiryReceiptBuilder.
     */
    public static MosaicExpiryReceiptBuilder create(final short version, final ReceiptTypeDto type, final MosaicIdDto artifactId) {
        return new MosaicExpiryReceiptBuilder(version, type, artifactId);
    }

    /**
     * Gets expiring mosaic id.
     *
     * @return Expiring mosaic id.
     */
    public MosaicIdDto getArtifactId() {
        return this.artifactId;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    @Override
    public int getSize() {
        int size = super.getSize();
        size += this.artifactId.getSize();
        return size;
    }

    /**
     * Creates an instance of MosaicExpiryReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MosaicExpiryReceiptBuilder.
     */
    public static MosaicExpiryReceiptBuilder loadFromBinary(final DataInputStream stream) {
        return new MosaicExpiryReceiptBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            final byte[] artifactIdBytes = this.artifactId.serialize();
            dataOutputStream.write(artifactIdBytes, 0, artifactIdBytes.length);
        });
    }
}
