/**
*** Copyright (c) 2016-present,
*** Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp. All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;

/** Binary layout for an inflation receipt. */
public final class InflationReceiptBuilder extends ReceiptBuilder implements Serializer {
    /** Mosaic. */
    private final MosaicBuilder mosaic;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected InflationReceiptBuilder(final DataInputStream stream) {
        super(stream);
        this.mosaic = MosaicBuilder.loadFromBinary(stream);
    }

    /**
     * Constructor.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param mosaic Mosaic.
     */
    protected InflationReceiptBuilder(final short version, final ReceiptTypeDto type, final MosaicBuilder mosaic) {
        super(version, type);
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        this.mosaic = mosaic;
    }

    /**
     * Creates an instance of InflationReceiptBuilder.
     *
     * @param version Receipt version.
     * @param type Receipt type.
     * @param mosaic Mosaic.
     * @return Instance of InflationReceiptBuilder.
     */
    public static InflationReceiptBuilder create(final short version, final ReceiptTypeDto type, final MosaicBuilder mosaic) {
        return new InflationReceiptBuilder(version, type, mosaic);
    }

    /**
     * Gets mosaic.
     *
     * @return Mosaic.
     */
    public MosaicBuilder getMosaic() {
        return this.mosaic;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    @Override
    public int getSize() {
        int size = super.getSize();
        size += this.mosaic.getSize();
        return size;
    }

    /**
     * Creates an instance of InflationReceiptBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of InflationReceiptBuilder.
     */
    public static InflationReceiptBuilder loadFromBinary(final DataInputStream stream) {
        return new InflationReceiptBuilder(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            final byte[] superBytes = super.serialize();
            dataOutputStream.write(superBytes, 0, superBytes.length);
            final byte[] mosaicBytes = this.mosaic.serialize();
            dataOutputStream.write(mosaicBytes, 0, mosaicBytes.length);
        });
    }
}
