/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;

/** A PublicKey used for voting during the [finalization process](/concepts/block.html#finalization).. */
public final class VotingPublicKeyDto implements Serializer {
    /** A PublicKey used for voting during the [finalization process](/concepts/block.html#finalization).. */
    private final ByteBuffer votingPublicKey;

    /**
     * Constructor.
     *
     * @param votingPublicKey A PublicKey used for voting during the [finalization process](/concepts/block.html#finalization)..
     */
    public VotingPublicKeyDto(final ByteBuffer votingPublicKey) {
        this.votingPublicKey = votingPublicKey;
    }

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize.
     */
    public VotingPublicKeyDto(final DataInputStream stream) {
        try {
            this.votingPublicKey = GeneratorUtils.readByteBuffer(stream, 32);
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Gets A PublicKey used for voting during the [finalization process](/concepts/block.html#finalization)..
     *
     * @return A PublicKey used for voting during the [finalization process](/concepts/block.html#finalization)..
     */
    public ByteBuffer getVotingPublicKey() {
        return this.votingPublicKey;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        return 32;
    }

    /**
     * Creates an instance of VotingPublicKeyDto from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of VotingPublicKeyDto.
     */
    public static VotingPublicKeyDto loadFromBinary(final DataInputStream stream) {
        return new VotingPublicKeyDto(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            dataOutputStream.write(this.votingPublicKey.array(), 0, this.votingPublicKey.array().length);
        });
    }
}

