/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Shared content between VotingKeyLinkTransaction and EmbeddedVotingKeyLinkTransaction.
**/
public class VotingKeyLinkTransactionBodyBuilder implements Serializer {

    /** Linked voting public key.. **/
    private final VotingPublicKeyDto linkedPublicKey;

    /** Starting finalization epoch.. **/
    private final FinalizationEpochDto startEpoch;

    /** Ending finalization epoch.. **/
    private final FinalizationEpochDto endEpoch;

    /** Account link action.. **/
    private final LinkActionDto linkAction;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected VotingKeyLinkTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.linkedPublicKey = VotingPublicKeyDto.loadFromBinary(stream);
            this.startEpoch = FinalizationEpochDto.loadFromBinary(stream);
            this.endEpoch = FinalizationEpochDto.loadFromBinary(stream);
            this.linkAction = LinkActionDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of VotingKeyLinkTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of VotingKeyLinkTransactionBodyBuilder.
     */
    public static VotingKeyLinkTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new VotingKeyLinkTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param linkedPublicKey Linked voting public key..
    * @param startEpoch Starting finalization epoch..
    * @param endEpoch Ending finalization epoch..
    * @param linkAction Account link action..
    */
    protected VotingKeyLinkTransactionBodyBuilder(VotingPublicKeyDto linkedPublicKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch, LinkActionDto linkAction) {
        GeneratorUtils.notNull(linkedPublicKey, "linkedPublicKey is null");
        GeneratorUtils.notNull(startEpoch, "startEpoch is null");
        GeneratorUtils.notNull(endEpoch, "endEpoch is null");
        GeneratorUtils.notNull(linkAction, "linkAction is null");
        this.linkedPublicKey = linkedPublicKey;
        this.startEpoch = startEpoch;
        this.endEpoch = endEpoch;
        this.linkAction = linkAction;
    }
    
    /**
     * Creates an instance of VotingKeyLinkTransactionBodyBuilder.
     *
     * @param linkedPublicKey Linked voting public key..
     * @param startEpoch Starting finalization epoch..
     * @param endEpoch Ending finalization epoch..
     * @param linkAction Account link action..
     * @return Instance of VotingKeyLinkTransactionBodyBuilder.
     */
    public static VotingKeyLinkTransactionBodyBuilder create(VotingPublicKeyDto linkedPublicKey, FinalizationEpochDto startEpoch, FinalizationEpochDto endEpoch, LinkActionDto linkAction) {
        return new VotingKeyLinkTransactionBodyBuilder(linkedPublicKey, startEpoch, endEpoch, linkAction);
    }

    /**
     * Gets Linked voting public key..
     *
     * @return Linked voting public key..
     */
    public VotingPublicKeyDto getLinkedPublicKey() {
        return this.linkedPublicKey;
    }

    /**
     * Gets Starting finalization epoch..
     *
     * @return Starting finalization epoch..
     */
    public FinalizationEpochDto getStartEpoch() {
        return this.startEpoch;
    }

    /**
     * Gets Ending finalization epoch..
     *
     * @return Ending finalization epoch..
     */
    public FinalizationEpochDto getEndEpoch() {
        return this.endEpoch;
    }

    /**
     * Gets Account link action..
     *
     * @return Account link action..
     */
    public LinkActionDto getLinkAction() {
        return this.linkAction;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.linkedPublicKey.getSize();
        size += this.startEpoch.getSize();
        size += this.endEpoch.getSize();
        size += this.linkAction.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.linkedPublicKey);
            GeneratorUtils.writeEntity(dataOutputStream, this.startEpoch);
            GeneratorUtils.writeEntity(dataOutputStream, this.endEpoch);
            GeneratorUtils.writeEntity(dataOutputStream, this.linkAction);
        });
    }
}

