/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Binary layout for a transaction
**/
public class TransactionBuilder implements Serializer {

    /** Entity size in bytes.
This size includes the header and the full payload of the entity. I.e, the size field matches the size reported in the structure documentation (plus the variable part, if there is any).. **/
    private int size;

    /** Reserved padding to align Signature to an 8-byte boundary.. **/
    private final int verifiableEntityHeaderReserved1;

    /** Entity's signature generated by the signing account.. **/
    private final SignatureDto signature;

    /** Public key of the signer of the entity.. **/
    private final PublicKeyDto signerPublicKey;

    /** Reserved padding to align end of EntityBody to an 8-byte boundary.. **/
    private final int entityBodyReserved1;

    /** Version of this structure.. **/
    private final byte version;

    /** Network on which this entity was created.. **/
    private final NetworkTypeDto network;

    /** Transaction type. **/
    private final TransactionTypeDto type;

    /** Transaction fee. **/
    private final AmountDto fee;

    /** Transaction deadline. **/
    private final TimestampDto deadline;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected TransactionBuilder(DataInputStream stream) {
        try {
            this.size = Integer.reverseBytes(stream.readInt());
            this.verifiableEntityHeaderReserved1 = Integer.reverseBytes(stream.readInt());
            this.signature = SignatureDto.loadFromBinary(stream);
            this.signerPublicKey = PublicKeyDto.loadFromBinary(stream);
            this.entityBodyReserved1 = Integer.reverseBytes(stream.readInt());
            this.version = stream.readByte();
            this.network = NetworkTypeDto.loadFromBinary(stream);
            this.type = TransactionTypeDto.loadFromBinary(stream);
            this.fee = AmountDto.loadFromBinary(stream);
            this.deadline = TimestampDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of TransactionBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of TransactionBuilder.
     */
    public static TransactionBuilder loadFromBinary(DataInputStream stream) {
        return new TransactionBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param signature Entity's signature generated by the signing account..
    * @param signerPublicKey Public key of the signer of the entity..
    * @param version Version of this structure..
    * @param network Network on which this entity was created..
    * @param type Transaction type.
    * @param fee Transaction fee.
    * @param deadline Transaction deadline.
    */
    protected TransactionBuilder(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, AmountDto fee, TimestampDto deadline) {
        GeneratorUtils.notNull(signature, "signature is null");
        GeneratorUtils.notNull(signerPublicKey, "signerPublicKey is null");
        GeneratorUtils.notNull(version, "version is null");
        GeneratorUtils.notNull(network, "network is null");
        GeneratorUtils.notNull(type, "type is null");
        GeneratorUtils.notNull(fee, "fee is null");
        GeneratorUtils.notNull(deadline, "deadline is null");
        this.verifiableEntityHeaderReserved1 = 0;
        this.signature = signature;
        this.signerPublicKey = signerPublicKey;
        this.entityBodyReserved1 = 0;
        this.version = version;
        this.network = network;
        this.type = type;
        this.fee = fee;
        this.deadline = deadline;
    }
    
    /**
     * Creates an instance of TransactionBuilder.
     *
     * @param signature Entity's signature generated by the signing account..
     * @param signerPublicKey Public key of the signer of the entity..
     * @param version Version of this structure..
     * @param network Network on which this entity was created..
     * @param type Transaction type.
     * @param fee Transaction fee.
     * @param deadline Transaction deadline.
     * @return Instance of TransactionBuilder.
     */
    public static TransactionBuilder create(SignatureDto signature, PublicKeyDto signerPublicKey, byte version, NetworkTypeDto network, TransactionTypeDto type, AmountDto fee, TimestampDto deadline) {
        return new TransactionBuilder(signature, signerPublicKey, version, network, type, fee, deadline);
    }

    /**
     * Gets Entity size in bytes.
This size includes the header and the full payload of the entity. I.e, the size field matches the size reported in the structure documentation (plus the variable part, if there is any)..
     *
     * @return Entity size in bytes.
This size includes the header and the full payload of the entity. I.e, the size field matches the size reported in the structure documentation (plus the variable part, if there is any)..
     */
    public int getStreamSize() {
        return this.size;
    }

    /**
     * Gets Reserved padding to align Signature to an 8-byte boundary..
     *
     * @return Reserved padding to align Signature to an 8-byte boundary..
     */
    private int getVerifiableEntityHeaderReserved1() {
        return this.verifiableEntityHeaderReserved1;
    }

    /**
     * Gets Entity's signature generated by the signing account..
     *
     * @return Entity's signature generated by the signing account..
     */
    public SignatureDto getSignature() {
        return this.signature;
    }

    /**
     * Gets Public key of the signer of the entity..
     *
     * @return Public key of the signer of the entity..
     */
    public PublicKeyDto getSignerPublicKey() {
        return this.signerPublicKey;
    }

    /**
     * Gets Reserved padding to align end of EntityBody to an 8-byte boundary..
     *
     * @return Reserved padding to align end of EntityBody to an 8-byte boundary..
     */
    private int getEntityBodyReserved1() {
        return this.entityBodyReserved1;
    }

    /**
     * Gets Version of this structure..
     *
     * @return Version of this structure..
     */
    public byte getVersion() {
        return this.version;
    }

    /**
     * Gets Network on which this entity was created..
     *
     * @return Network on which this entity was created..
     */
    public NetworkTypeDto getNetwork() {
        return this.network;
    }

    /**
     * Gets transaction type.
     *
     * @return Transaction type.
     */
    public TransactionTypeDto getType() {
        return this.type;
    }

    /**
     * Gets transaction fee.
     *
     * @return Transaction fee.
     */
    public AmountDto getFee() {
        return this.fee;
    }

    /**
     * Gets transaction deadline.
     *
     * @return Transaction deadline.
     */
    public TimestampDto getDeadline() {
        return this.deadline;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 4; // size
        size += 4; // verifiableEntityHeaderReserved1
        size += this.signature.getSize();
        size += this.signerPublicKey.getSize();
        size += 4; // entityBodyReserved1
        size += 1; // version
        size += this.network.getSize();
        size += this.type.getSize();
        size += this.fee.getSize();
        size += this.deadline.getSize();
        return size;
    }


    /**
     * Gets the body builder of the object.
     *
     * @return Body builder.
     */
    public Serializer getBody() {
        return null;
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getSize()));
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getVerifiableEntityHeaderReserved1()));
            GeneratorUtils.writeEntity(dataOutputStream, this.signature);
            GeneratorUtils.writeEntity(dataOutputStream, this.signerPublicKey);
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getEntityBodyReserved1()));
            dataOutputStream.writeByte((byte) this.getVersion());
            GeneratorUtils.writeEntity(dataOutputStream, this.network);
            GeneratorUtils.writeEntity(dataOutputStream, this.type);
            GeneratorUtils.writeEntity(dataOutputStream, this.fee);
            GeneratorUtils.writeEntity(dataOutputStream, this.deadline);
        });
    }
}

