/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Shared content between SecretLockTransaction and EmbeddedSecretLockTransaction.
**/
public class SecretLockTransactionBodyBuilder implements Serializer {

    /** Address that receives the funds once successfully unlocked by a SecretProofTransaction.. **/
    private final UnresolvedAddressDto recipientAddress;

    /** Hashed proof.. **/
    private final Hash256Dto secret;

    /** Locked mosaics.. **/
    private final UnresolvedMosaicBuilder mosaic;

    /** Number of blocks to wait for the SecretProofTransaction.. **/
    private final BlockDurationDto duration;

    /** Algorithm used to hash the proof.. **/
    private final LockHashAlgorithmDto hashAlgorithm;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected SecretLockTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.recipientAddress = UnresolvedAddressDto.loadFromBinary(stream);
            this.secret = Hash256Dto.loadFromBinary(stream);
            this.mosaic = UnresolvedMosaicBuilder.loadFromBinary(stream);
            this.duration = BlockDurationDto.loadFromBinary(stream);
            this.hashAlgorithm = LockHashAlgorithmDto.loadFromBinary(stream);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of SecretLockTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of SecretLockTransactionBodyBuilder.
     */
    public static SecretLockTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new SecretLockTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param recipientAddress Address that receives the funds once successfully unlocked by a SecretProofTransaction..
    * @param secret Hashed proof..
    * @param mosaic Locked mosaics..
    * @param duration Number of blocks to wait for the SecretProofTransaction..
    * @param hashAlgorithm Algorithm used to hash the proof..
    */
    protected SecretLockTransactionBodyBuilder(UnresolvedAddressDto recipientAddress, Hash256Dto secret, UnresolvedMosaicBuilder mosaic, BlockDurationDto duration, LockHashAlgorithmDto hashAlgorithm) {
        GeneratorUtils.notNull(recipientAddress, "recipientAddress is null");
        GeneratorUtils.notNull(secret, "secret is null");
        GeneratorUtils.notNull(mosaic, "mosaic is null");
        GeneratorUtils.notNull(duration, "duration is null");
        GeneratorUtils.notNull(hashAlgorithm, "hashAlgorithm is null");
        this.recipientAddress = recipientAddress;
        this.secret = secret;
        this.mosaic = mosaic;
        this.duration = duration;
        this.hashAlgorithm = hashAlgorithm;
    }
    
    /**
     * Creates an instance of SecretLockTransactionBodyBuilder.
     *
     * @param recipientAddress Address that receives the funds once successfully unlocked by a SecretProofTransaction..
     * @param secret Hashed proof..
     * @param mosaic Locked mosaics..
     * @param duration Number of blocks to wait for the SecretProofTransaction..
     * @param hashAlgorithm Algorithm used to hash the proof..
     * @return Instance of SecretLockTransactionBodyBuilder.
     */
    public static SecretLockTransactionBodyBuilder create(UnresolvedAddressDto recipientAddress, Hash256Dto secret, UnresolvedMosaicBuilder mosaic, BlockDurationDto duration, LockHashAlgorithmDto hashAlgorithm) {
        return new SecretLockTransactionBodyBuilder(recipientAddress, secret, mosaic, duration, hashAlgorithm);
    }

    /**
     * Gets Address that receives the funds once successfully unlocked by a SecretProofTransaction..
     *
     * @return Address that receives the funds once successfully unlocked by a SecretProofTransaction..
     */
    public UnresolvedAddressDto getRecipientAddress() {
        return this.recipientAddress;
    }

    /**
     * Gets Hashed proof..
     *
     * @return Hashed proof..
     */
    public Hash256Dto getSecret() {
        return this.secret;
    }

    /**
     * Gets Locked mosaics..
     *
     * @return Locked mosaics..
     */
    public UnresolvedMosaicBuilder getMosaic() {
        return this.mosaic;
    }

    /**
     * Gets Number of blocks to wait for the SecretProofTransaction..
     *
     * @return Number of blocks to wait for the SecretProofTransaction..
     */
    public BlockDurationDto getDuration() {
        return this.duration;
    }

    /**
     * Gets Algorithm used to hash the proof..
     *
     * @return Algorithm used to hash the proof..
     */
    public LockHashAlgorithmDto getHashAlgorithm() {
        return this.hashAlgorithm;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += this.recipientAddress.getSize();
        size += this.secret.getSize();
        size += this.mosaic.getSize();
        size += this.duration.getSize();
        size += this.hashAlgorithm.getSize();
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            GeneratorUtils.writeEntity(dataOutputStream, this.recipientAddress);
            GeneratorUtils.writeEntity(dataOutputStream, this.secret);
            GeneratorUtils.writeEntity(dataOutputStream, this.mosaic);
            GeneratorUtils.writeEntity(dataOutputStream, this.duration);
            GeneratorUtils.writeEntity(dataOutputStream, this.hashAlgorithm);
        });
    }
}

