/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;

/** A 32-byte (256 bit) integer derived from a private key.
It serves as the public identifier of the [key pair](/concepts/cryptography.html#key-pair) and can be disseminated widely. It is used to prove that an entity was signed with the paired private key.. */
public final class PublicKeyDto implements Serializer {
    /** A 32-byte (256 bit) integer derived from a private key.
It serves as the public identifier of the [key pair](/concepts/cryptography.html#key-pair) and can be disseminated widely. It is used to prove that an entity was signed with the paired private key.. */
    private final ByteBuffer publicKey;

    /**
     * Constructor.
     *
     * @param publicKey A 32-byte (256 bit) integer derived from a private key.
It serves as the public identifier of the [key pair](/concepts/cryptography.html#key-pair) and can be disseminated widely. It is used to prove that an entity was signed with the paired private key..
     */
    public PublicKeyDto(final ByteBuffer publicKey) {
        this.publicKey = publicKey;
    }

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize.
     */
    public PublicKeyDto(final DataInputStream stream) {
        try {
            this.publicKey = GeneratorUtils.readByteBuffer(stream, 32);
        } catch(Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Gets A 32-byte (256 bit) integer derived from a private key.
It serves as the public identifier of the [key pair](/concepts/cryptography.html#key-pair) and can be disseminated widely. It is used to prove that an entity was signed with the paired private key..
     *
     * @return A 32-byte (256 bit) integer derived from a private key.
It serves as the public identifier of the [key pair](/concepts/cryptography.html#key-pair) and can be disseminated widely. It is used to prove that an entity was signed with the paired private key..
     */
    public ByteBuffer getPublicKey() {
        return this.publicKey;
    }

    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        return 32;
    }

    /**
     * Creates an instance of PublicKeyDto from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of PublicKeyDto.
     */
    public static PublicKeyDto loadFromBinary(final DataInputStream stream) {
        return new PublicKeyDto(stream);
    }

    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize(dataOutputStream -> {
            dataOutputStream.write(this.publicKey.array(), 0, this.publicKey.array().length);
        });
    }
}

