/**
*** Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
*** Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.
*** All rights reserved.
***
*** This file is part of Catapult.
***
*** Catapult is free software: you can redistribute it and/or modify
*** it under the terms of the GNU Lesser General Public License as published by
*** the Free Software Foundation, either version 3 of the License, or
*** (at your option) any later version.
***
*** Catapult is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*** GNU Lesser General Public License for more details.
***
*** You should have received a copy of the GNU Lesser General Public License
*** along with Catapult. If not, see <http://www.gnu.org/licenses/>.
**/

package io.nem.symbol.catapult.builders;

import java.io.DataInputStream;
import java.nio.ByteBuffer;
import java.util.EnumSet;
import java.util.List;

/**
* Shared content between MultisigAccountModificationTransaction and EmbeddedMultisigAccountModificationTransaction.
**/
public class MultisigAccountModificationTransactionBodyBuilder implements Serializer {

    /** Relative change to the **minimum** number of cosignatures required when **removing a cosignatory**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**.. **/
    private final byte minRemovalDelta;

    /** Relative change to the **minimum** number of cosignatures required when **approving a transaction**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**.. **/
    private final byte minApprovalDelta;

    /** Reserved padding to align addressAdditions to an 8-byte boundary.. **/
    private final int multisigAccountModificationTransactionBodyReserved1;

    /** Cosignatory address additions.
All accounts in this list will be able to cosign transactions on behalf of the multisig account. The number of required cosignatures depends on the configured minimum approval and minimum removal values.. **/
    private final List<UnresolvedAddressDto> addressAdditions;

    /** Cosignatory address deletions.
All accounts in this list will stop being able to cosign transactions on behalf of the multisig account. A transaction containing **any** address in this array requires a number of cosignatures at least equal to the minimum removal value.. **/
    private final List<UnresolvedAddressDto> addressDeletions;

    /**
     * Constructor - Creates an object from stream.
     *
     * @param stream Byte stream to use to serialize the object.
     */
    protected MultisigAccountModificationTransactionBodyBuilder(DataInputStream stream) {
        try {
            this.minRemovalDelta = stream.readByte();
            this.minApprovalDelta = stream.readByte();
            final byte addressAdditionsCount = stream.readByte();
            final byte addressDeletionsCount = stream.readByte();
            this.multisigAccountModificationTransactionBodyReserved1 = Integer.reverseBytes(stream.readInt());
            this.addressAdditions = GeneratorUtils.loadFromBinaryArray(UnresolvedAddressDto::loadFromBinary, stream, addressAdditionsCount, 0);
            this.addressDeletions = GeneratorUtils.loadFromBinaryArray(UnresolvedAddressDto::loadFromBinary, stream, addressDeletionsCount, 0);
        } catch (Exception e) {
            throw GeneratorUtils.getExceptionToPropagate(e);
        }
    }

    /**
     * Creates an instance of MultisigAccountModificationTransactionBodyBuilder from a stream.
     *
     * @param stream Byte stream to use to serialize the object.
     * @return Instance of MultisigAccountModificationTransactionBodyBuilder.
     */
    public static MultisigAccountModificationTransactionBodyBuilder loadFromBinary(DataInputStream stream) {
        return new MultisigAccountModificationTransactionBodyBuilder(stream);
    }
    
    /**
    * Constructor.
    *
    * @param minRemovalDelta Relative change to the **minimum** number of cosignatures required when **removing a cosignatory**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
    * @param minApprovalDelta Relative change to the **minimum** number of cosignatures required when **approving a transaction**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
    * @param addressAdditions Cosignatory address additions.
All accounts in this list will be able to cosign transactions on behalf of the multisig account. The number of required cosignatures depends on the configured minimum approval and minimum removal values..
    * @param addressDeletions Cosignatory address deletions.
All accounts in this list will stop being able to cosign transactions on behalf of the multisig account. A transaction containing **any** address in this array requires a number of cosignatures at least equal to the minimum removal value..
    */
    protected MultisigAccountModificationTransactionBodyBuilder(byte minRemovalDelta, byte minApprovalDelta, List<UnresolvedAddressDto> addressAdditions, List<UnresolvedAddressDto> addressDeletions) {
        GeneratorUtils.notNull(minRemovalDelta, "minRemovalDelta is null");
        GeneratorUtils.notNull(minApprovalDelta, "minApprovalDelta is null");
        GeneratorUtils.notNull(addressAdditions, "addressAdditions is null");
        GeneratorUtils.notNull(addressDeletions, "addressDeletions is null");
        this.minRemovalDelta = minRemovalDelta;
        this.minApprovalDelta = minApprovalDelta;
        this.multisigAccountModificationTransactionBodyReserved1 = 0;
        this.addressAdditions = addressAdditions;
        this.addressDeletions = addressDeletions;
    }
    
    /**
     * Creates an instance of MultisigAccountModificationTransactionBodyBuilder.
     *
     * @param minRemovalDelta Relative change to the **minimum** number of cosignatures required when **removing a cosignatory**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     * @param minApprovalDelta Relative change to the **minimum** number of cosignatures required when **approving a transaction**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     * @param addressAdditions Cosignatory address additions.
All accounts in this list will be able to cosign transactions on behalf of the multisig account. The number of required cosignatures depends on the configured minimum approval and minimum removal values..
     * @param addressDeletions Cosignatory address deletions.
All accounts in this list will stop being able to cosign transactions on behalf of the multisig account. A transaction containing **any** address in this array requires a number of cosignatures at least equal to the minimum removal value..
     * @return Instance of MultisigAccountModificationTransactionBodyBuilder.
     */
    public static MultisigAccountModificationTransactionBodyBuilder create(byte minRemovalDelta, byte minApprovalDelta, List<UnresolvedAddressDto> addressAdditions, List<UnresolvedAddressDto> addressDeletions) {
        return new MultisigAccountModificationTransactionBodyBuilder(minRemovalDelta, minApprovalDelta, addressAdditions, addressDeletions);
    }

    /**
     * Gets Relative change to the **minimum** number of cosignatures required when **removing a cosignatory**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     *
     * @return Relative change to the **minimum** number of cosignatures required when **removing a cosignatory**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     */
    public byte getMinRemovalDelta() {
        return this.minRemovalDelta;
    }

    /**
     * Gets Relative change to the **minimum** number of cosignatures required when **approving a transaction**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     *
     * @return Relative change to the **minimum** number of cosignatures required when **approving a transaction**.
E.g., when moving from 0 to 2 cosignatures this number would be **2**. When moving from 4 to 3 cosignatures, the number would be **-1**..
     */
    public byte getMinApprovalDelta() {
        return this.minApprovalDelta;
    }

    /**
     * Gets Reserved padding to align addressAdditions to an 8-byte boundary..
     *
     * @return Reserved padding to align addressAdditions to an 8-byte boundary..
     */
    private int getMultisigAccountModificationTransactionBodyReserved1() {
        return this.multisigAccountModificationTransactionBodyReserved1;
    }

    /**
     * Gets Cosignatory address additions.
All accounts in this list will be able to cosign transactions on behalf of the multisig account. The number of required cosignatures depends on the configured minimum approval and minimum removal values..
     *
     * @return Cosignatory address additions.
All accounts in this list will be able to cosign transactions on behalf of the multisig account. The number of required cosignatures depends on the configured minimum approval and minimum removal values..
     */
    public List<UnresolvedAddressDto> getAddressAdditions() {
        return this.addressAdditions;
    }

    /**
     * Gets Cosignatory address deletions.
All accounts in this list will stop being able to cosign transactions on behalf of the multisig account. A transaction containing **any** address in this array requires a number of cosignatures at least equal to the minimum removal value..
     *
     * @return Cosignatory address deletions.
All accounts in this list will stop being able to cosign transactions on behalf of the multisig account. A transaction containing **any** address in this array requires a number of cosignatures at least equal to the minimum removal value..
     */
    public List<UnresolvedAddressDto> getAddressDeletions() {
        return this.addressDeletions;
    }


    /**
     * Gets the size of the object.
     *
     * @return Size in bytes.
     */
    public int getSize() {
        int size = 0;
        size += 1; // minRemovalDelta
        size += 1; // minApprovalDelta
        size += 1; // addressAdditionsCount
        size += 1; // addressDeletionsCount
        size += 4; // multisigAccountModificationTransactionBodyReserved1
        size +=  GeneratorUtils.getSumSize(this.addressAdditions, 0);
        size +=  GeneratorUtils.getSumSize(this.addressDeletions, 0);
        return size;
    }



    /**
     * Serializes an object to bytes.
     *
     * @return Serialized bytes.
     */
    public byte[] serialize() {
        return GeneratorUtils.serialize((dataOutputStream) -> {
            dataOutputStream.writeByte((byte) this.getMinRemovalDelta());
            dataOutputStream.writeByte((byte) this.getMinApprovalDelta());
            dataOutputStream.writeByte((byte) GeneratorUtils.getSize(this.getAddressAdditions()));
            dataOutputStream.writeByte((byte) GeneratorUtils.getSize(this.getAddressDeletions()));
            dataOutputStream.writeInt(Integer.reverseBytes((int) this.getMultisigAccountModificationTransactionBodyReserved1()));
            GeneratorUtils.writeList(dataOutputStream, this.addressAdditions, 0);
            GeneratorUtils.writeList(dataOutputStream, this.addressDeletions, 0);
        });
    }
}

